import re
import uuid

from django_yauth.util import get_real_ip

from .consts import CollabLoggerConfig
from .utils import apply_mask


def _get_request_id(request):
    return (
        request.META.get('HTTP_X_REQ_ID', None)
        or request.META.get('HTTP_X_REQUEST_ID', None)
        or 'auto-' + uuid.uuid4().hex
    )


def _get_headers(request, cfg: CollabLoggerConfig):
    for header, mask_type in cfg.headers.items():
        value = request.META.get(header)
        if value is not None:
            yield header, apply_mask(mask_type, value)


def extract_normalized_route(request):
    if request.resolver_match:
        if hasattr(request.resolver_match, 'route'):
            return re.sub(r'\(\?P<(.*)>.*\)\??', '<\\1>', request.resolver_match.route)
        else:
            return request.resolver_match.func.__name__

    return None


def build_request(request, cfg: CollabLoggerConfig):
    return {
        'id': _get_request_id(request),
        'method': request.method,
        'path': request.get_full_path(),
        'headers': dict(_get_headers(request, cfg)),
    }


def build_auth(request, cfg: CollabLoggerConfig):
    ctx = {'type': 'guest', 'client_ip': get_real_ip(request)}

    if getattr(request, 'yauser', None) is None or not request.yauser.is_authenticated():
        return ctx

    ctx.update(
        {
            'uid': request.yauser.uid,
            'login': request.yauser.get_username(),
        }
    )

    if not request.yauser.authenticated_by:
        ctx['type'] = 'unknown'
        return ctx

    ctx['type'] = request.yauser.authenticated_by.mechanism_name

    try:
        if request.yauser.authenticated_by.mechanism_name == 'oauth':
            ctx['application'] = request.yauser.blackbox_result.oauth.client_name
        if request.yauser.authenticated_by.mechanism_name == 'tvm':
            ctx['application'] = request.yauser.service_ticket.src
    except Exception:  # noqa
        pass

    return ctx
