# http://www.djangosnippets.org/snippets/383/
# Source: http://djangosnippets.org/snippets/1495/
import base64
import logging

from django.db import models
from django.db.models.signals import post_init
from django.utils.text import compress_string

logger = logging.getLogger(__name__)


def uncompress_string(s):
    """
    Helper function to reverse django.utils.text.compress_string
    """
    from io import BytesIO
    import gzip

    try:
        temp_val = s.encode('ascii')
        val = base64.b64decode(temp_val)
        # Если на вход функции попал unicode с русским текстом, то
        # декодирование из base64 вернет пустую строку, чего не очень-то
        # и хочется
        if temp_val and not val:
            raise ValueError

        zbuf = BytesIO(val)
        zfile = gzip.GzipFile(fileobj=zbuf)
        ret = zfile.read()
        zfile.close()
    except Exception:
        logger.exception('Unable decompress string')
        ret = s
    return ret


class CompressedTextField(models.TextField):
    """
    Transparently compress data before hitting the db
    and uncompress after fetching
    """

    def get_db_prep_save(self, value, connection):
        if value is not None:
            if isinstance(value, str):
                value = value.encode('utf8')
            value = compress_string(value)
            value = base64.b64encode(value).decode('ascii')
        return models.TextField.get_db_prep_save(self, value, connection=connection)

    def _get_val_from_obj(self, obj):
        if obj:
            value = uncompress_string(getattr(obj, self.attname))
            if value is not None:
                try:
                    value = value.decode('utf8')
                except (UnicodeDecodeError, AttributeError):
                    pass
                return value
            else:
                return self.get_default()
        else:
            return self.get_default()

    def post_init(self, instance=None, **kwargs):
        value = self._get_val_from_obj(instance)
        setattr(instance, self.attname, value)

    def contribute_to_class(self, cls, name):
        super(CompressedTextField, self).contribute_to_class(cls, name)
        post_init.connect(self.post_init, sender=cls)

    def get_internal_type(self):
        return "TextField"

    def db_type(self, connection):
        return 'text'

    def south_field_triple(self):
        """Returns a suitable description of this field for South."""
        # We'll just introspect the _actual_ field.
        from south.modelsinspector import introspector

        field_class = "django.db.models.fields.TextField"
        args, kwargs = introspector(self)
        # That's our definition!
        return (field_class, args, kwargs)
