import celery
import logging

from django.db import transaction
from ylog.context import log_context

from django_tools_log_context.celery import CtxAwareMixin
from django_tools_log_context.profiler import execution_profiler


log = logging.getLogger(__name__)


class ABCCtxAwareMixin(CtxAwareMixin):
    _execution_threshold = 0

    def __call__(self, *args, **kwargs):
        log_context_data = getattr(self.request, 'log_context', None)
        if log_context_data is None:
            headers = self.request.headers or {}
            log_context_data = headers.get('log_context', {})
        log_context_data['celery_task'] = {'name': self.name}
        with log_context(**log_context_data):
            with execution_profiler(self.name, code_block_type='celery task', threshold=self._execution_threshold):
                log.info('Started celery task %s', self.name)
                # Из-за того, что у CtxAwareMixin определен метод __call__ вызывается не тот метод
                # в celery.app.trace.build_tracer: fun = task if task_has_custom(task, '__call__') else task.run
                # но нам не нужно вызывать __call__ у таски, там делается self.push_request()
                # и на request_stack попадает пустой объект Context()
                # вызовем явно метод run
                return self.run(*args, **kwargs)


class AbcCeleryTask(ABCCtxAwareMixin, celery.Task):
    def delay_on_commit(self, *args, **kwargs):
        return transaction.on_commit(lambda: self.delay(*args, **kwargs))

    def apply_async_on_commit(self, *args, **kwargs):
        return transaction.on_commit(lambda: self.apply_async(*args, **kwargs))
