from collections import OrderedDict

from django.core.paginator import Paginator as DjangoPaginator
from django.utils.functional import cached_property
from django.utils.six.moves.urllib import parse as urlparse

from rest_framework import pagination
from rest_framework.response import Response
from rest_framework.utils.urls import remove_query_param

from plan.api.filters import (
    BooleanFilter,
    ServiceDescendantsFilterSet,
    ServiceWithDescendantsFilter,
    RoleIncludeGlobalFilter
)
from plan.api.mixins import OrderingMixin, SelectiveFieldsMixin
from plan.api.serializers import (
    AttributesSerializer,
    ModelSerializer,
    CompactStaffSerializer,
    CompactDepartmentSerializer,
    ContactTypeSerializer,
    CompactServiceSerializer,
    CompactRoleScopeSerializer,
    CompactRoleSerializer,
)


class ABCCursorPagination(pagination.CursorPagination):
    page_size = 500
    page_size_query_param = 'page_size'
    max_page_size = 1000

    def get_ordering(self, request, queryset, view):
        # Если запрос без фильтров, то сортируем только по id
        # здесь queryset ещё не отфильтрован по данным из cursor
        if not queryset.query.where:
            return getattr(view, 'default_cursor_ordering', ('id', ),)
        return super(ABCCursorPagination, self).get_ordering(request, queryset, view)

    def get_paginated_response(self, data):
        return Response(OrderedDict([
            ('next', self.get_next_link()),
            ('previous', self.get_previous_link()),
            ('results', data)
        ]))

    def get_page_size(self, request):
        # ToDo: обновить drf
        # Копипаста метода из djangorestframework==3.9.0
        if self.page_size_query_param:
            try:
                return pagination._positive_int(
                    request.query_params[self.page_size_query_param],
                    strict=True,
                    cutoff=self.max_page_size
                )
            except (KeyError, ValueError):
                pass

        return self.page_size

    def decode_cursor(self, request):
        """
        Given a request with a cursor, return a `Cursor` instance.
        """
        # Determine if we have a cursor, and if so then decode it.
        encoded = request.query_params.get(self.cursor_query_param)
        if encoded is None:
            return None

        try:

            querystring = encoded.rsplit('-', 1)
            tokens = {'p': querystring[0], 'r': querystring[1]}

            offset = 0
            offset = pagination._positive_int(offset, cutoff=self.offset_cutoff)

            reverse = tokens['r']
            reverse = bool(int(reverse))

            position = tokens['p']
        except (TypeError, ValueError):
            raise pagination.NotFound(self.invalid_cursor_message)

        return pagination.Cursor(offset=offset, reverse=reverse, position=position)

    def encode_cursor(self, cursor):
        """
        Given a Cursor instance, return an url with encoded cursor.
        """
        tokens = {'o': 0, 'r': 0, 'p': 0}
        if cursor.offset != 0:
            tokens['o'] = str(cursor.offset)
        if cursor.reverse:
            tokens['r'] = '1'
        if cursor.position is not None:
            tokens['p'] = cursor.position

        querystring = '-'.join(str(tokens[key]) for key in ['p', 'r'])
        encoded = querystring
        return pagination.replace_query_param(self.base_url, self.cursor_query_param, encoded)


class ABCPaginator(DjangoPaginator):
    @cached_property
    def count(self):
        """
        Копия базового метода, но переопределён object_list.
        Это нужно, если в пагинатор передано queryset.nested_values(), иначе count делается от запроса с лишними JOIN
        """
        object_list = getattr(self.object_list, '_true_query', self.object_list)
        try:
            return object_list.count()
        except (AttributeError, TypeError):
            return len(object_list)


class ABCPagination(pagination.PageNumberPagination):
    page_size = 20
    page_size_query_param = 'page_size'
    django_paginator_class = ABCPaginator
    max_page_size = 1000

    def get_paginated_response(self, data):
        return Response(OrderedDict([
            ('count', self.page.paginator.count),
            ('next', self.get_next_link()),
            ('previous', self.get_previous_link()),
            ('total_pages', self.page.paginator.num_pages),
            ('results', data)
        ]))

    def get_next_link(self):
        if not self.page.has_next():
            return None
        url = self.request.build_absolute_uri()
        page_number = self.page.next_page_number()
        return self.replace_query_param(url, self.page_query_param, page_number)

    def get_previous_link(self):
        if not self.page.has_previous():
            return None
        url = self.request.build_absolute_uri()
        page_number = self.page.previous_page_number()
        if page_number == 1:
            return remove_query_param(url, self.page_query_param)
        return self.replace_query_param(url, self.page_query_param, page_number)

    @staticmethod
    def replace_query_param(url, key, val):
        (scheme, netloc, path, query, fragment) = urlparse.urlsplit(url)
        query_dict = urlparse.parse_qs(query, keep_blank_values=True)
        query_dict[key] = [val]
        # TODO: в этом месте желательно отсортировать дикт перед тем, как
        # пилить из него query-string
        # но здесь на вход идет уже закодированый юникод,
        # а sorted() его не прожевывает
        # надо придумать, как обрабатывать, чтоб urlencode пришел
        # уже закодированый юникод
        query = urlparse.urlencode(list(query_dict.items()), doseq=True)
        return urlparse.urlunsplit((scheme, netloc, path, query, fragment))


class ABCLargePagination(ABCPagination):
    max_page_size = 10000


class ABCLargeCursorPagination(ABCCursorPagination):
    max_page_size = 10000


# compatibility shimming
# TODO: нужно везде, где юзается from plan.api import base
# начать прямо импортировать конкретно нужные объекты
__all__ = [
    # base
    'ABCCursorPagination',
    "ABCPagination",
    # filters
    "BooleanFilter",
    "ServiceDescendantsFilterSet",
    "ServiceWithDescendantsFilter",
    "RoleIncludeGlobalFilter",
    # mixins
    "OrderingMixin",
    "SelectiveFieldsMixin",
    # serializers
    "AttributesSerializer",
    "ModelSerializer",
    "CompactStaffSerializer",
    "CompactDepartmentSerializer",
    "ContactTypeSerializer",
    "CompactServiceSerializer",
    "CompactRoleScopeSerializer",
    "CompactRoleSerializer",
]
