from requests import codes as http_codes

from rest_framework.exceptions import APIException


class ABCAPIException(APIException):
    '''
    Кастомный класс эксепшенов для DRF-апи ABC.
    Отличается от родного класса DRF тем, что не допускает в detail
    диктов или списков (в DRF ValidationError по умолчанию форсит туда
    итератор с ошибками формы) и содержит дополнительные поля
    для человекочитаемых ошибок.

    Новые классы ошибок нужно создавать, наследуясь от этого,
    при этом определяя в них:

    status_code - обязательно; http код, с которым будет отдаваться ошибка

    default_detail - обязательно;
                     сообщение ошибки, предназначенное для логов и дебага;
                     при отсутствии человекочитаемого текста, фронт отобразит
                     этот

    default_code - обязательно;
                   стринг с системным кодом ошибки;
                   фронт полагается на него для выбора кастомного шаблона

    default_title - необязательно;
                    дикт вида {ru: , en: } с человекочитаемыми заголовками

    default_message - необязательно;
                      дикт вида {ru: , en: } с человекочитаемыми сообщениями
    '''

    def __init__(self, detail=None, code=None, status_code=None, title=None, message=None, extra=None):
        if detail is None:
            self.detail = self.default_detail
        elif isinstance(detail, (list, dict)):
            # ValidationError в DRF может прислать в detail список или дикт
            # с парами {поле: [сообщение, код]}.
            # Мы это хотим отдавать не в detail, а отдельным полем.
            self.detail = self.default_detail
            self.extra = detail
        else:
            self.detail = detail

        self.code = code or self.default_code

        if status_code:
            self.status_code = status_code

        if title is None:
            if hasattr(self, 'default_title'):
                self.title = self.default_title
        else:
            self.title = title

        if message is None:
            if hasattr(self, 'default_message'):
                self.message = self.default_message
        else:
            self.message = message

        if extra is not None:
            # тут важно обратить внимание на проверку detail выше
            self.extra = extra


class BadRequest(ABCAPIException):
    status_code = http_codes.bad_request
    default_detail = 'Sent data is wrong.'
    default_code = 'bad_request'


class Unauthorized(ABCAPIException):
    status_code = http_codes.unauthorized
    default_detail = "You don't seem to be authorized."
    default_code = 'unauthorized'
    default_title = {
        'ru': 'Кажется, вы не авторизованы.',
        'en': "You do not seem to be authorized. "
    }
    default_message = {
        'ru': "Либо вы делаете запрос анонимно, либо что-то пошло не так.",
        'en': "Either you are making the request anonymously, or something went wrong."
    }


class NotFound(ABCAPIException):
    status_code = http_codes.not_found
    default_detail = 'Not found.'
    default_code = 'not_found'
    default_title = {
        'ru': 'Не найдено',
        'en': 'Not found'
    }
    default_message = {
        'ru': 'Наши еноты ничего не нашли по вашему запросу',
        'en': 'Our raccoons could not find anything to satisfy your query'
    }


class PermissionDenied(ABCAPIException):
    status_code = http_codes.forbidden
    default_detail = 'Permission denied.'
    default_code = 'forbidden'
    default_title = {
        'ru': 'Доступ запрещен',
        'en': 'Permission denied'
    }
    default_message = {
        'ru': 'Извините, вам сюда нельзя',
        'en': "Sorry, we can't let you in here"
    }


class Conflict(ABCAPIException):
    status_code = http_codes.conflict
    default_detail = 'Conflict in data.'
    default_code = 'conflict'
    default_title = {
        'ru': 'Конфликт в данных',
        'en': 'Conflict in data'
    }
    default_message = {
        'ru': 'Ваш запрос не может быть осуществлен из-за его конфликта с состоянием сервиса',
        'en': 'Your request cannot be fulfilled because it conflicts with service state'
    }


class IntegrationError(ABCAPIException):
    status_code = http_codes.internal_server_error
    default_detail = 'Integrated service did not fulfil the request.'
    default_code = 'integration_error'


class ValidationError(ABCAPIException):
    status_code = http_codes.bad_request
    default_detail = 'Validation error.'
    default_code = 'validation_error'
    default_title = {
        'ru': 'Ошибка валидации',
        'en': 'Validation error'
    }
    default_message = {
        'ru': 'Ваш запрос не прошел валидацию',
        'en': 'Your request did not pass validation',
    }


class ResourceTypeDisabled(BadRequest):
    default_code = 'resource_type_disabled'
    default_detail = 'Resource type is disabled.'
    default_message = {
        'ru': 'Мы не можем запросить ресурс этого типа, потому что его поставщик сейчас отключен.',
        'en': 'We cannot request a resource of this type because its supplier is currently disabled.',
    }


class ResourceRequestNoRoles(PermissionDenied):
    default_code = 'resource_request_denied'
    default_detail = 'Must have eligible role.'
    default_message = {
        'ru': 'Чтобы запросить этот ресурс, у вас должна быть соответствующая роль в сервисе-потребителе.',
        'en': 'You don\'t have a role in service-consumer that can request this resource.',
    }


class ResourceEditNoRoles(PermissionDenied):
    default_code = 'resource_edit_denied'
    default_detail = 'Must have eligible role.'
    default_message = {
        'ru': 'У вас нет роли в сервисе-потребителе, которая может редактировать этот ресурс.',
        'en': 'You don\'t have a role in this service-consumer that can edit this resource.',
    }


class ResourceRequestResponsibleOnly(PermissionDenied):
    default_code = 'resource_request_denied'
    default_detail = 'Must be service responsible.'
    default_message = {
        'ru': 'Запросить ресурсы этого типа для сервиса-потребителя могут только его управляющие.',
        'en': 'Only people responsible for this service-consumer can request resources of this type for it.',
    }


class ResourceEditResponsibleOnly(PermissionDenied):
    default_code = 'resource_edit_denied'
    default_detail = 'Must be service responsible.'
    default_message = {
        'ru': 'Вы не можете редактировать этот ресурс. Вы должны быть управляющим сервиса-потребителя.',
        'en': 'You can\'t edit this resource. You must be one of this service\'s responsibles.',
    }


class ResourceTypeWithoutEdit(BadRequest):
    default_code = 'resource_edit_disabled'
    default_detail = 'Resource type does not support editing.'
    default_message = {
        'ru': 'Поставщик ресурсов этого типа не поддерживает их редактирование.',
        'en': 'The supplier of resources of this type does not support editing them.'
    }


class ServiceNotExportable(BadRequest):
    default_code = 'service_not_exportable'
    default_detail = 'Service does not support operations.'
    default_message = {
        'ru': 'Сервис находится в песочнице, поэтому операции с ним запрещены.',
        'en': 'The service is in sandbox, so operations with them are prohibited.'
    }


class TooManyServices(BadRequest):
    default_code = 'too_many_services'
    default_detail = 'Too many services'
    default_message = {
        'ru': 'Слишком много сервисов. Сверните дерево или поменяйте фильтры',
        'en': 'Services count is too big. Reduce the tree or change filters',
    }


class ServiceTagsDuplicatingError(ValidationError):
    default_code = 'servicetags_duplicating'
    default_detail = 'Service Tags is duplicated'
    default_message = {
        'ru': 'У одного из предков используется несколько градиентных тегов.',
        'en': 'One or more ancestors have more than one gradient tags.',
    }


class GradientStructureError(ValidationError):
    default_code = 'gradient_structure'
    default_detail = 'The gradient structure is broken'
    default_message = {
        'ru': 'Градиентная структура нарушена.',
        'en': 'The gradient structure is broken.',
    }


class FailedDependency(ABCAPIException):
    status_code = http_codes.failed_dependency
    default_detail = 'The request failed because it depended on another request and that request failed.'
    default_code = 'failed_dependency'
