import json

from django.core.exceptions import PermissionDenied as Django403
from django.http import Http404 as Django404
from django import http

from rest_framework.views import set_rollback
from rest_framework.exceptions import APIException
from rest_framework.status import is_success

from plan.api.exceptions import ABCAPIException, NotFound, PermissionDenied
from plan.exceptions import Error as DeprecatedError
from plan.lib.views.mixins import JSONResponseMixin


# Убрать, когда откажемся от старого апи
def json_response(data, status, headers={}):
    content = json.dumps(
        data,
        default=JSONResponseMixin.serialize_unserializable,
        ensure_ascii=False,
    )
    return http.HttpResponse(
        content=content,
        status=status,
        content_type='application/json; charset=utf-8',
        **headers
    )


def abc_exception_handler(exc, context):
    """
    Any unhandled exceptions may return `None`, which will cause a 500 error
    to be raised.
    """
    if isinstance(exc, DeprecatedError):
        data = {
            'code': exc.error_code.lower(),
            'detail': getattr(exc, 'message', str(exc)),
        }
        set_rollback()
        return json_response({'error': data}, status=exc.http_code)

    if isinstance(exc, (ABCAPIException, APIException)):
        headers = {}
        if getattr(exc, 'auth_header', None):
            headers['WWW-Authenticate'] = exc.auth_header
        if getattr(exc, 'wait', None):
            headers['Retry-After'] = '%d' % exc.wait

        if isinstance(exc, ABCAPIException):
            data = {
                'detail': exc.detail,
                'code': exc.code,
            }
        else:
            data = {
                'detail': exc.detail,
                'code': exc.default_code,
            }

        if hasattr(exc, 'message'):
            data['message'] = exc.message

        if hasattr(exc, 'title'):
            data['title'] = exc.title

        if isinstance(exc.detail, (list, dict)):
            data['detail'] = exc.default_detail
            data['extra'] = exc.detail

        if hasattr(exc, 'extra'):
            data['extra'] = exc.extra

        set_rollback()
        if not is_success(exc.status_code):
            data = {'error': data}
        return json_response(data, status=exc.status_code, headers=headers)

    if isinstance(exc, (Django403, Django404)):

        if isinstance(exc, Django403):
            exc = PermissionDenied()
        else:
            exc = NotFound()

        data = {
            'detail': exc.detail,
            'code': exc.code,
            'title': exc.title,
            'message': getattr(exc, 'message', str(exc)),
        }

        set_rollback()
        return json_response({'error': data}, status=exc.status_code)

    return None


def old_abc_exception_handler(exc, context):
    data = {
        'content': {},
        'error': {
            'message': context['message'],
            'code': context['code'],
            'params': {}
        }
    }

    return json_response(data, status=exc.status_code)
