import collections.abc
import pytz

import django_filters
from django.core.exceptions import (
    ObjectDoesNotExist,
    ValidationError,
)
from django.conf import settings
from django.utils.translation import ugettext_lazy as _
from rest_framework import serializers
from rest_framework.utils import html
from rest_framework.fields import is_simple_callable, DateTimeField

from plan.api import validators
from plan.services.models import Service
from plan.api.validators import ServiceNameValidator


def get_attribute(instance, attrs):
    """
    Метод из версии drf 3.6.4, в более поздних при None не происходит
    возврата значения, а происходит AttributeError, что нам не везде подходит
    """
    for attr in attrs:
        if instance is None:
            # Break out early if we get `None` at any point in a nested lookup.
            return None
        try:
            if isinstance(instance, collections.abc.Mapping):
                instance = instance[attr]
            else:
                instance = getattr(instance, attr)
        except ObjectDoesNotExist:
            return None
        if is_simple_callable(instance):
            try:
                instance = instance()
            except (AttributeError, KeyError) as exc:
                # If we raised an Attribute or KeyError here it'd get treated
                # as an omitted field in `Field.get_attribute()`. Instead we
                # raise a ValueError to ensure the exception is not masked.
                raise ValueError('Exception raised in callable attribute "{0}"; original exception was: {1}'.format(attr, exc))

    return instance


class IntegerInterfacedSerializerField(serializers.PrimaryKeyRelatedField):
    '''
    Этот филд работает, как PrimaryKeyRelatedField на вход, но на выдачу отдает
    сериализацию объектов указанным сериалайзером.
    Он нам нужен, чтоб с фронта при редактировании связанных с основной
    сущностей могли присылать только список с id.
    '''

    def __init__(self, **kwargs):
        self.serializer = kwargs.pop('serializer', None)
        super(IntegerInterfacedSerializerField, self).__init__(**kwargs)

    def get_choices(self, cutoff=None):
        return {}

    def use_pk_only_optimization(self):
        return False

    def get_serializer(self, value):
        serializer = self.serializer(value, context=self.context)
        serializer.parent = self.parent
        serializer.field_name = self.field_name
        serializer.queryset = self.queryset
        return serializer

    def to_representation(self, value):
        if self.serializer is not None:
            serializer = self.get_serializer(value)
            return serializer.data
        else:
            return super(IntegerInterfacedSerializerField, self).to_representation(value)


class SlugInterfacedSerializerField(serializers.SlugRelatedField):
    def __init__(self, **kwargs):
        self.serializer = kwargs.pop('serializer', None)
        super(SlugInterfacedSerializerField, self).__init__(**kwargs)

    def to_representation(self, value):
        if self.serializer is not None:
            serializer = self.serializer(value, context=self.context)
            serializer.parent = self.parent
            serializer.field_name = self.field_name
            return serializer.data
        else:
            return super(SlugInterfacedSerializerField, self).to_representation(value)


class MappingField(serializers.DictField):
    def __init__(self, mapping, *args, **kwargs):
        self.mapping = mapping

        kwargs['source'] = '*'

        super(MappingField, self).__init__(*args, **kwargs)

    def to_representation(self, obj):
        ret = {}
        if obj is not None:
            for key, field in self.mapping.items():
                ret[key] = get_attribute(obj, field.split('__'))

        return ret

    def _get_serializer_fields(self):
        return self.parent.fields

    def to_internal_value(self, data):
        if html.is_html_input(data):
            data = html.parse_html_dict(data)

        if not isinstance(data, dict):
            self.fail('not_a_dict', input_type=type(data).__name__)

        ret = {}

        for key, field in self.mapping.items():
            if self.required and (key not in data or not data[key]):
                self.fail('required')

            elif key not in data:
                continue
            serializer_fields = self._get_serializer_fields()
            if key in serializer_fields:
                data[key] = serializer_fields[key].run_validation(data[key])

            ret[field] = data[key]
        return ret


class LocaleServiceNameField(MappingField):
    ru = serializers.CharField(
        source='name_ru',
        validators=[
            validators.UniqueFieldValidator(
                queryset=Service.objects.alive(),
                field='name',
                lookup='iexact',
                message=_('Имя сервиса должно быть уникальным'),
            ),
            ServiceNameValidator(),
        ]
    )
    en = serializers.CharField(
        source='name_en',
        validators=[
            validators.UniqueFieldValidator(
                queryset=Service.objects.alive(),
                field='name_en',
                lookup='iexact',
                message=_('Английское имя сервиса должно быть уникальным'),
            ),
            ServiceNameValidator(),
        ]
    )

    def __init__(self, mapping, *args, **kwargs):
        super().__init__(mapping, *args, **kwargs)
        self.en.bind(field_name='', parent=self)
        self.ru.bind(field_name='', parent=self)

    def _get_serializer_fields(self):
        fields = {
            'ru': self.ru,
            'en': self.en,
        }
        fields.update(super()._get_serializer_fields())
        return fields


class CustomPrimaryKeyRelatedField(serializers.PrimaryKeyRelatedField):
    """
    Класс, который на вход может принимать число или объект класса модели
    """
    def to_representation(self, value):
        data = value if isinstance(value, int) else value.pk
        if self.pk_field is not None:
            return self.pk_field.to_representation(data)
        return data


class CustomModelChoiceField(django_filters.fields.ModelChoiceField):
    """
    Переопределяем метод to_python из django 1.11 - в связи с изменением
    логики в новой версии django_filter - тут теперь проверяется при фильтрации
    что такое значение есть в базе - в общем случае это нам не нужно так как
      - приводит к 400 вместо пустого результата если значения в базе нет
    """
    def to_python(self, value):
        if value in self.empty_values:
            return None

        try:
            key = self.to_field_name or 'pk'
            value = self.queryset.get(**{key: value})
        except (ValueError, TypeError):
            raise ValidationError(self.error_messages['invalid_choice'], code='invalid_choice')
        except self.queryset.model.DoesNotExist:
            value = settings.NULL_VALUE
        return value


class CustomModelMultipleChoiceField(django_filters.fields.ModelMultipleChoiceField):
    def _check_values(self, value):
        """
        Метод скопирован из django 1.11 у ModelMultipleChoiceField с небольшими изменениями
        дело в том что в новой версии django_filter изменилось поведение - теперь проверяется
        валидность django form полученной из полей сериализатора.
        А django в этом типе поля кидает ошибку валидации если параметра по которому мы фильтуем
        нет в базе вообще, в результате будет не пустой ответ, а ошибка валидации

        Поэтому в этом методе кусок кода про проверку существования таких объектов убран
        """
        null = self.null_label is not None and value and self.null_value in value
        if null:  # remove the null value and any potential duplicates
            value = [v for v in value if v != self.null_value]

        key = self.to_field_name or 'pk'

        try:
            value = frozenset(value)
        except TypeError:
            # list of lists isn't hashable, for example
            raise ValidationError(
                self.error_messages['list'],
                code='list',
            )
        try:
            qs = self.queryset.filter(**{'%s__in' % key: value})
        except (ValueError, TypeError):
            raise ValidationError(
                self.error_messages['invalid_pk_value'],
                code='invalid_pk_value',
                params={'pk': value},
            )

        result = list(qs)
        result += [self.null_value] if null else []
        if not result:
            return settings.NULL_VALUE
        return result


class CustomDateField(DateTimeField):
    def default_timezone(self):
        return pytz.timezone('UTC')
