import django_filters
from django.core.exceptions import ImproperlyConfigured
from django.db.models import (
    ManyToManyField,
    Q,
    ForeignKey,
    BooleanField,
)
from django.forms import ValidationError as FormsValidationError
from django.forms.fields import MultipleChoiceField
from django.conf import settings
from rest_framework.filters import OrderingFilter, BaseFilterBackend

from plan.api.exceptions import BadRequest, ValidationError
from plan.services.models import Service
from plan.services.state import SERVICE_STATE
from plan.api.fields import CustomModelMultipleChoiceField, CustomModelChoiceField


class ModelFilterMixin:
    def filter(self, qs, value):
        """
        Изменяем тут дефолтное поведение - если по параметру фильтрации не нашлось
        ничего - в новой версии django_filter по умолчанию возвращаются все значения,
        а нам нужно возвращать пустой ответ
        """
        # обрабатываем фильтр по __in
        if value and isinstance(value, list):
            value = [element for element in value if element != settings.NULL_VALUE]
            if not value:
                return qs.none()

        if value is settings.NULL_VALUE:
            return qs.none()

        return super().filter(qs, value)


class IntegerListFilter(django_filters.Filter):
    def filter(self, qs, value):
        if value in (None, ''):
            return qs
        integers = [int(v) for v in value.split(',') if v.isdigit()]
        return qs.filter(**{f'{self.field_name}__{self.lookup_expr}': integers})


class CustomBooleanFilter(django_filters.rest_framework.BooleanFilter):
    def filter(self, queryset, value):
        if value in ('1', 'True', 'true'):
            value = True
        elif value in ('0', 'False', 'false'):
            value = False
        return super().filter(queryset, value)


class CustomModelChoiceFilter(ModelFilterMixin, django_filters.filters.ModelChoiceFilter):
    field_class = CustomModelChoiceField


class CustomModelMultipleChoiceFilter(ModelFilterMixin, django_filters.ModelMultipleChoiceFilter):
    field_class = CustomModelMultipleChoiceField


FILTER_FOR_DBFIELD_DEFAULTS = django_filters.rest_framework.filterset.FILTER_FOR_DBFIELD_DEFAULTS
FILTER_FOR_DBFIELD_DEFAULTS[ForeignKey]['filter_class'] = CustomModelChoiceFilter
FILTER_FOR_DBFIELD_DEFAULTS[ManyToManyField]['filter_class'] = CustomModelMultipleChoiceFilter
FILTER_FOR_DBFIELD_DEFAULTS[BooleanField]['filter_class'] = CustomBooleanFilter


class PlanFilterSet(django_filters.rest_framework.filterset.FilterSet):
    FILTER_DEFAULTS = FILTER_FOR_DBFIELD_DEFAULTS

    @property
    def qs(self):
        try:
            return super().qs
        except FormsValidationError as exc:
            raise ValidationError(extra=exc.message_dict)


class BaseListFilter(BaseFilterBackend):
    query_param = None
    field = None

    def get_values(self, request):
        query_list = request.GET.getlist(self.query_param)
        return self.filter_values(query_list)

    def filter_values(self, values):

        output = set()

        for value in values:
            if value == 'none':
                output.add(None)

            else:
                try:
                    value = int(value)
                    if value > 0:
                        output.add(value)

                except Exception:
                    pass

        return list(output) if output else None

    def filter_queryset(self, request, queryset, view):

        if None in (self.query_param, self.field):
            raise ImproperlyConfigured('Query parameter and field must be set')

        values = self.get_values(request)

        if values is None:
            return queryset

        if values:
            condition = Q(**{f'{self.field}__in': [_f for _f in values if _f]})

            if None in values:
                condition |= Q(**{f'{self.field}__isnull': True})

        else:
            condition = Q()

        return self.apply_filter(queryset, condition)

    def apply_filter(self, queryset, condition):

        qs = queryset.filter(condition)

        field = queryset.model._meta.get_field(self.field)

        # Prevent from having duplicated instances when filtering by
        # many-to-many fields.
        if isinstance(field, ManyToManyField):
            qs = qs.distinct()

        return qs


class IncludeFilter(BaseListFilter):
    query_param = 'id'
    field = 'id'


class ExcludeFilter(BaseListFilter):
    query_param = 'exclude_id'
    field = 'id'

    def apply_filter(self, queryset, cond):
        return super(ExcludeFilter, self).apply_filter(queryset, ~cond)


class StaffIdIncludeFilter(IncludeFilter):
    query_param = 'id'
    field = 'staff_id'


class StaffIdExcludeFilter(ExcludeFilter):
    query_param = 'exclude_id'
    field = 'staff_id'


class CharInFilter(django_filters.BaseInFilter, django_filters.CharFilter):
    pass


class BooleanFilter(django_filters.Filter):
    def filter(self, queryset, value):
        method = self.get_method(queryset)

        if value in ('1', 'True', 'true'):
            return method(**{self.field_name: True})
        elif value in ('0', 'False', 'false'):
            return method(**{self.field_name: False})
        else:
            return queryset


class OrderingFilterBackend(OrderingFilter):
    def remove_invalid_fields(self, queryset, fields, view, request):
        valid_fields = [item[0] for item in self.get_valid_fields(queryset, view, {'request': request})]

        difference = {field.lstrip('-') for field in fields} - set(valid_fields)
        if difference:
            raise BadRequest(
                detail='api handle does not support sorting by fields: {}'.format(
                    ', '.join(difference)
                )
            )

        return super(OrderingFilterBackend, self).remove_invalid_fields(queryset, fields, view, request)


class ServiceDescendantsFilterSet(PlanFilterSet):
    service = CustomModelMultipleChoiceFilter(
        field_name='service',
        queryset=Service.objects.all(),
        lookup_expr='in',
        method='filter_services',
        distinct=False,
    )

    service__in = IntegerListFilter(field_name='service', lookup_expr='in')

    descendants = django_filters.BooleanFilter(method='filter_descendants')

    def filter_services(self, queryset, name, value):
        if value == settings.NULL_VALUE:
            return queryset.none()

        if not value:
            return queryset

        # наверное есть более красивый вариант, я пока не нашел
        if self.data.get('descendants'):
            value = list(
                s
                for service in value
                for s in service.get_descendants(include_self=True)
            )
        return queryset.filter(service__in=value)

    def filter_descendants(self, queryset, name, value):
        return queryset


class ServiceWithDescendantsFilter(CustomModelMultipleChoiceFilter):
    def __init__(self, *args, **kwargs):
        self.only_active = kwargs.pop('only_active', False)
        self.only_id = kwargs.pop('only_id', False)
        super(ServiceWithDescendantsFilter, self).__init__(*args, **kwargs)

    def use_inheritance_settings(self):
        inheritance_settings = self.parent.data.get('use_inheritance_settings')

        if inheritance_settings in ('1', 'True', 'true'):
            return True

        return False

    def filter(self, qs, value):
        if value and value != settings.NULL_VALUE:
            services_pks = [obj.pk for obj in value]
            if not self.use_inheritance_settings():
                parents = Service._closure_model.objects.filter(parent__in=services_pks).values_list('child')
                all_descendants = Service.objects.filter(Q(pk__in=parents) | Q(pk__in=services_pks))
            else:
                all_descendants_with_settings = []
                for service in value:
                    descendants = service.get_descendants(include_self=True)
                    services_with_inheritance_false = descendants.filter(
                        ~Q(pk=service.pk),
                        membership_inheritance=False
                    )
                    children_to_exclude = service._closure_model.objects.filter(
                        ~Q(child=service),
                        parent__in=services_with_inheritance_false,
                    ).values_list('child')
                    descendants = descendants.exclude(
                        ~Q(pk=service.pk),
                        Q(pk__in=services_with_inheritance_false) | Q(pk__in=children_to_exclude)
                    ).values_list('id', flat=True)

                    all_descendants_with_settings.extend(descendants)

                all_descendants = Service.objects.filter(pk__in=all_descendants_with_settings)

            if self.only_active:
                all_descendants = all_descendants.exclude(~Q(state__in=SERVICE_STATE.ACTIVE_STATES) & ~Q(pk__in=services_pks))

            if self.only_id:
                value = all_descendants.values_list('id', flat=True)
            else:
                value = all_descendants

        return super(ServiceWithDescendantsFilter, self).filter(qs, value)


class RoleIncludeGlobalFilter(CustomModelMultipleChoiceFilter):
    def use_include_global(self):
        include_global = self.parent.data.get('include_global')

        if include_global in ('1', 'True', 'true'):
            return True

        return False

    def filter(self, qs, value):
        if not value:
            return super(RoleIncludeGlobalFilter, self).filter(qs, value)

        if self.use_include_global():
            return qs.filter(Q(service__isnull=True) | Q(service__in=value))

        return super(RoleIncludeGlobalFilter, self).filter(qs, value)


class NoValidationChoiceField(MultipleChoiceField):
    def valid_value(self, value):
        return True


class NoValidationMultipleChoiceFilter(django_filters.MultipleChoiceFilter):
    field_class = NoValidationChoiceField
