"""
Все функции в этом модуле переопределяются в crowdtest-окружении.
"""
import datetime
import inspect
import logging
import sys
from itertools import chain
from typing import Optional
from django.conf import settings

from plan.api import exceptions
from plan.api.idm.helpers import crowdtest_disabled, get_role_by_membership
from plan.idm import nodes, exceptions as idm_exceptions
from plan.idm.adapters import RoleManager, RoleRequestManager, ApproveRequestManager
from plan.idm.constants import INACTIVE_STATES, ACTIVE_STATES
from plan.idm.manager import idm_manager, Manager
from plan.roles.models import Role
from plan.services.models import Service, ServiceMember
from plan.staff.models import Department, Staff

log = logging.getLogger(__name__)

LIMIT = 1000


def add_service(service: Service):
    log.info('Adding service %s to IDM', service)

    service_nodes = (
        nodes.get_service_node(service),
        nodes.get_service_roles_node(service),
    )
    manager = idm_manager()
    nodes_to_create = list(nodes.find_missing_nodes(service_nodes, manager=manager))
    role_nodes = nodes.get_service_role_nodes(service)
    role_nodes_to_create = list(nodes.find_missing_nodes(role_nodes, manager=manager))

    last_exception = None
    for node in chain(nodes_to_create, role_nodes_to_create):
        try:
            node.register(manager)

        except idm_exceptions.IDMError as exc:
            logging.info('Cannot create a node %s.', node.slug_path)
            last_exception = exc

    if last_exception is not None:
        raise last_exception


def assert_service_node_exists(service: Service, parent: Service) -> None:
    log.info('Check node for %s exists in IDM', service)

    parent_node = nodes.get_service_node(parent)
    if parent is None or parent_node.exists():
        node = nodes.build_service_node(service, parent_node)
        if node.exists():
            return

    raise idm_exceptions.IDMError('Node for service %s not found.', service)


def add_service_head(service: Service, staff: Staff, requester: Staff = None, comment: str = ''):
    log.info('Set head <%s> for service %s', staff.login, service)

    role = Role.objects.globalwide().get(code=settings.ABC_DEFAULT_HEAD_ROLE)

    manager = idm_manager()
    RoleRequestManager.request(
        manager=manager,
        system=settings.ABC_IDM_SYSTEM_SLUG,
        user=staff.login,
        path=nodes.get_role_node(service, role).value_path,
        comment='Инициатор - %s (%s). ' % (requester.get_full_name(), requester.login) + comment
    )


def rename_service(service: Service, name: str, name_en: str):
    log.info('Rename %s to <%s> (<%s>)', service, name, name_en)
    node = nodes.get_service_node(service)
    node.update(name={'ru': name, 'en': name_en})


def move_service(service: Service, new_parent: Service):
    log.info('Move service %s to %s', service, new_parent)

    service_node = nodes.get_service_node(service)
    if not service_node.exists():  # если сервис уже перенесен в new_parent
        assert_service_node_exists(service, new_parent)
    else:
        new_parent_node = nodes.get_service_node(new_parent)
        service_node.update(parent=new_parent_node.key_path)


def delete_service(service: Service, fail_if_already_deleted: bool = False):
    log.info('Delete service %s', service)
    node = nodes.get_service_node(service)
    if node.exists():
        node.delete()
    elif fail_if_already_deleted:
        raise exceptions.IntegrationError(
            detail="Tried to delete a service note that doesn't exist in IDM.",
            extra={'slug_path': node.slug_path}
        )


def set_review_policy_to_service(service: Service, review_required: bool, max_retries: int = 2):
    log.info('Set %s review policy to all roles of %s', service, review_required)
    for role_node in nodes.get_service_role_nodes(service):
        retry_counter = max_retries
        while retry_counter:
            try:
                role_node.update(review_required=review_required)
                break
            except idm_exceptions.IDMError:
                log.debug('Request to IDM failed with error', exc_info=True)
                retry_counter -= 1
            log.warning(f'Error occurred during update review policy role node {role_node}')


def check_review_policy_of_service_roles(service: Service):
    manager = idm_manager()

    for role_node in nodes.get_service_role_nodes(service):
        try:
            node_data = role_node.fetch_data(manager=manager)
            if node_data['review_required'] is not service.review_required:
                role_node.update(manager=manager, review_required=service.review_required)
        except idm_exceptions.IDMError:
            log.error(f'Error occurred during update review policy role node {role_node}')
            raise


def add_role(custom_role: Role):
    log.info('Add custom role %s to %s', custom_role.name, custom_role.service)
    node = nodes.get_role_node(custom_role.service, custom_role)
    node.register()


def delete_role(custom_role: Role):
    assert custom_role.service is not None
    log.info('Delete custom role %s from %s', custom_role.name, custom_role.service)
    node = nodes.get_role_node(custom_role.service, custom_role)
    node.delete()


def get_roles(
        service: Service,
        state: str = 'granted',
        full: bool = False,
        staff: Staff = None,
        requester: Staff = None,
        manager: Manager = None,
        timeout: Optional[int] = None,
        retry: Optional[int] = None,
) -> list:
    log.info('Get roles: service %s, state %s', service, state)

    if not manager:
        manager = idm_manager(timeout=timeout, retry=retry)

    params = {
        'system': settings.ABC_IDM_SYSTEM_SLUG,
        'path': nodes.get_service_roles_node(service).value_path,
        'state': state if isinstance(state, str) else ','.join(state),
        '_requester': requester and requester.login or settings.ABC_ZOMBIK_LOGIN,
    }

    if staff:
        params['user'] = staff.login

    roles = RoleManager.get_roles(manager, params, full)

    return roles['objects']


def get_roles_for_service(
        service: Service,
        offset: Optional[int] = 0,
        limit: Optional[int] = settings.ABC_IDM_LIMIT,
        states: Optional[list] = None,
        requester: Staff = None,
        manager: Manager = None,
        timeout: Optional[int] = None,
        retry: Optional[int] = None,
) -> dict:
    log.info(f'Get roles for service: service {service.slug}')

    if not manager:
        manager = idm_manager(timeout=timeout, retry=retry)

    params = {
        'abc_slug': service.slug,
        'limit': limit,
        '_requester': requester and requester.login or settings.ABC_ZOMBIK_LOGIN,
    }
    if states:
        params['state'] = ','.join(states)

    return RoleManager.get_roles(manager, params, no_meta=False, offset=offset)


def get_roles_with_permissions(service: Service, requester: Staff = None) -> list:
    """
        В интерфейсе для отображения кнопок нужны permissions из ручки ролей. IDM отдает их только при запросе
        отдельной роли. Получение всего списка ролей по одной очень тормозит (TOOLSUP-17632)
        Поэтому здесь мы идем на упрощение. Для granted ролей разрешения считаем сами, для остальных получаем из IDM,
        предполагая, что их значительно меньше.
        Если IDM вынесет permissions в список ролей, можно будет выкинуть этот костыль.
    """
    granted = get_roles(service, state='granted', requester=requester)

    states = INACTIVE_STATES + ACTIVE_STATES
    states.remove('granted')

    try:
        other = get_roles(service, state=states, full=True, requester=requester)
    except idm_exceptions.BadRequest:  # Нет запрошенных ролей
        other = []

    return granted + other


def request_membership(
        service: Service,
        subject,
        role: Role,
        comment: str = '',
        requester: Staff = None,
        deprive_after: int = None,
        deprive_at: datetime.datetime = None,
        simulate: bool = False,
        silent: bool = False,
        timeout: Optional[int] = None,
        retry: Optional[int] = None,
):
    log.info('Request membership: service %s, subject %s, role %s', service, subject, role)

    manager = idm_manager(timeout=timeout, retry=retry)

    params = {
        'manager': manager,
        'system': settings.ABC_IDM_SYSTEM_SLUG,
        'path': nodes.get_role_node(service, role).value_path,
        'comment': 'Роль запрошена через ABC. ' + comment,
        '_requester': requester and requester.login or settings.ABC_ZOMBIK_LOGIN,
    }

    if simulate:
        params['simulate'] = True

    if silent:
        params['silent'] = True

    if deprive_at and deprive_after:
        pass
    elif deprive_at:
        params['deprive_at'] = deprive_at.isoformat()
    elif deprive_after:
        params['deprive_after_days'] = deprive_after

    if isinstance(subject, Staff):
        params['user'] = subject.login

    elif isinstance(subject, Department):
        params['group'] = subject.staff_id

    else:
        raise ValueError('Bad subject type: %s' % type(subject))

    return RoleRequestManager.request(**params)


def approve_role(
    role_id: int, requester: Staff,
    timeout: Optional[int] = None,
    retry: Optional[int] = None
):
    _resolve_request(role_id, requester, True, timeout=timeout, retry=retry)


def decline_role(
    role_id: int, requester: Staff,
    timeout: Optional[int] = None,
    retry: Optional[int] = None
):
    _resolve_request(role_id, requester, False, timeout=timeout, retry=retry)


def _resolve_request(
    role_id: int, requester: Staff,
    approve: bool,
    timeout: Optional[int] = None,
    retry: Optional[int] = None
):
    action = 'approve' if approve else 'decline'
    log.info('%s role: staff %s, role %s', action.capitalize(), requester.login, role_id)

    manager = idm_manager(timeout=timeout, retry=retry)

    role = RoleManager.get_role(manager, role_id, _requester=requester.login)

    if role.get('approve_request'):
        getattr(ApproveRequestManager, action)(manager, role['approve_request']['id'], requester.login)

    else:
        log.info('Approve request not found')


def deprive_role(
    member: ServiceMember, comment: str = '',
    requester: Staff = None,
    timeout: Optional[int] = None,
    retry: Optional[int] = None
):
    log.info('Deprive role: service %s, subject %s, role %s', member.service, member.subject, member.role.pk)

    manager = idm_manager(timeout=timeout, retry=retry)

    params = {
        'system': settings.ABC_IDM_SYSTEM_SLUG,
        'path': nodes.get_role_node(member.service, member.role).value_path,
        'type': 'active',
        '_requester': requester and requester.login or settings.ABC_ZOMBIK_LOGIN,

    }

    if isinstance(member.subject, Staff):
        params['user'] = member.subject.login
        params['ownership'] = 'personal'

    elif isinstance(member.subject, Department):
        params['group'] = member.subject.staff_id

    else:
        raise ValueError('Bad subject type: %s' % type(member.subject))

    roles = RoleManager.get_roles(manager, params)

    if not roles['objects']:
        member.deprive()  # отзываем роль на своей стороне, если не нашли такую в IDM
        log.warning(
            'Could not find role for depriving: service %s, subject %s, role %s',
            member.service, member.subject, member.role.pk
        )
        return

    idm_role_id = roles['objects'][0]['id']

    deprive_params = {
        'comment': comment or 'Роль отозвана из ABC',
        '_requester': params['_requester']
    }
    RoleManager.deprive(manager, idm_role_id, **deprive_params)


def rerequest_role(
    service_member: ServiceMember,
    requester: Staff = None,
    timeout: Optional[int] = None,
    retry: Optional[int] = None,
):
    log.info('Rerequest role %s', service_member)

    idm_role = get_role_by_membership(service_member)

    manager = idm_manager(timeout=timeout, retry=retry)
    RoleManager.rerequest(
        manager,
        idm_role['id'],
        _requester=requester.login if requester else idm_role['role_request']['requester']['username'],
        comment='Перезапрос из ABC',
    )


def rerequest_requested(service: Service):
    """
    Пересчитывает ответственных в запрошенных ролях
    """
    log.info('Rerequest roles in service %s', service.slug)

    manager = idm_manager()
    roles = get_roles(service, state='requested', manager=manager, full=True)

    with manager:
        for role in roles:
            RoleManager.request(
                manager,
                role['id'],
                _requester=role['role_request']['requester']['username'],
                comment='В связи с изменением состава менеджеров команды',
            )

    if not manager.batch_result.ok:
        raise exceptions.IntegrationError(extra=manager.batch_result.failed)


def get_chown_requests(service: Service, full: bool = True, **params) -> list:
    owner_role = Role.get_exclusive_owner()

    roles = get_roles(service, full=full, state=INACTIVE_STATES, **params)
    roles = [
        role
        for role in roles
        if int(role['node']['data']['role']) == owner_role.pk
    ]

    return roles


def approve_owner_role(service: Service, staff: Staff, requester: Staff):
    _resolve_owner_role(service, staff, requester, True)


def decline_owner_role(service: Service, staff: Staff, requester: Staff):
    _resolve_owner_role(service, staff, requester, False)


def _resolve_owner_role(service: Service, staff: Staff, requester: Staff, approve: bool):
    roles = get_chown_requests(service, staff=staff)

    if len(roles) == 1:
        role_id = roles[0]['id']
        _resolve_request(role_id, requester, approve)

    else:
        raise ValueError('No owner role for service %s staff %s', service.slug, staff.login)


def delete_requests(service: Service):
    log.info('Delete all requests in service %s', service.slug)

    roles = get_roles(service, state='requested')
    manager = idm_manager()
    comment = 'Сервис закрыт.'

    with manager:
        for role in roles:
            RoleManager.deprive(manager, role['id'], comment=comment)

    if not manager.batch_result.ok:
        raise exceptions.IntegrationError(extra=manager.batch_result.failed)


def delete_chown_requests(service: Service, current_owner_login: str):
    log.info('Delete chown requests in service %s', service.slug)

    roles = get_chown_requests(service, full=False)
    manager = idm_manager()
    comment = '''Руководителем сервиса выбран другой сотрудник (%s).
        Обратитесь к руководителю вышестоящего сервиса, если вы считаете это решение неудачным.''' % current_owner_login

    with manager:
        for role in roles:
            if role['user']['username'] != current_owner_login:
                RoleManager.deprive(manager, role['id'], comment=comment)

    if not manager.batch_result.ok:
        raise exceptions.IntegrationError(extra=manager.batch_result.failed)


if settings.CROWDTEST:
    for member_name, member in inspect.getmembers(sys.modules[__name__]):
        if (inspect.isfunction(member) and member.__module__ == __name__):
            globals()[member_name] = crowdtest_disabled(member)
