from django import forms
from django.conf import settings
from django.contrib.auth.models import ContentType, Permission
from django.db.models import Q
from django_idm_api.validation import GetRolesForm

from plan.api.idm.fields import NoValidationField, UserField, GroupField
from plan.resources.models import Resource
from plan.roles.models import Role
from plan.services.constants.permissions import (
    SERVICE_TAG_APP,
    SERVICE_TAG_MODEL,
    CHANGE_SERVICE_TAG_CODENAME,
    CHANGE_SERVICE_TAG_TEMPLATE,
)
from plan.services.models import Service


class ErrorMixin(object):
    def get_errors(self):
        errors = {
            field: '\n'.join(errors_list)
            for field, errors_list in self.errors.items()
        }
        return errors

    def get_error_message(self):
        messages = []
        for fieldname, error_list in self.errors.items():
            if fieldname == '__all__':
                messages.append(', '.join(error_list))
            else:
                label = self.fields[fieldname].label
                messages.append('%s: %s' % (label, ', '.join(error_list)))
        message = '; '.join(messages)
        return message


class BaseForm(forms.Form, ErrorMixin):
    """Форма, умеющая по-разному работать со списком ошибок"""


class AddRoleForm(BaseForm):

    _FULL_INTERNAL_ROLE_LIST = dict(settings.ABC_INTERNAL_ROLES + settings.ABC_INTERNAL_ACCESS_ROLES)
    role_data = NoValidationField(label='Роль')
    fields = NoValidationField(label='Поля', required=False)

    def clean_role_data(self):
        role_data = self.cleaned_data['role_data']

        if role_data.get('type') not in ('internal', 'services', 'service_tags'):
            raise forms.ValidationError('Unknown role type in role data')

        if role_data['type'] == 'internal':
            role_name = role_data['internal_key']
            if role_name not in self._FULL_INTERNAL_ROLE_LIST:
                raise forms.ValidationError('Unknown internal role')

            self.cleaned_data['role'] = role_name

        elif role_data['type'] == 'service_tags':
            service_tag_slug = role_data['service_tags_key']
            try:
                Permission.objects.get(
                    codename=CHANGE_SERVICE_TAG_CODENAME % service_tag_slug,
                    content_type=ContentType.objects.get(app_label=SERVICE_TAG_APP, model=SERVICE_TAG_MODEL)
                )
            except Permission.DoesNotExist:
                raise forms.ValidationError(
                    'Permission for service tag {0} does not exist'.format(service_tag_slug)
                )
            self.cleaned_data['role'] = CHANGE_SERVICE_TAG_TEMPLATE % service_tag_slug

        else:
            try:
                self.cleaned_data['service'] = self.find_service(role_data)
            except ValueError:
                raise forms.ValidationError('Cannot find service slug in role data')

            try:
                self.cleaned_data['role'] = (
                    Role.objects
                    .filter(Q(service=None) | Q(service=self.cleaned_data['service']))
                    .get(pk=role_data['role'])
                )
            except KeyError:
                raise forms.ValidationError('Cannot find role scope in role_data {0}'.format(role_data))
            except Role.DoesNotExist:
                raise forms.ValidationError(
                    'Cannot find role {0} in role_data {1}'.format(role_data['role'], role_data)
                )

        return role_data

    def find_service(self, role_data):
        # к сожалению, IDM присылает не путь, а словарь,
        # нам надо найти в нем ключевую ноду самого нижнего сервиса

        for key, value in role_data.items():
            if value == '*':
                # звёздочка указывает на нижний уровень
                # ключ на этом уровне выглядит как "{service_slug}_key".
                # отрежем "_key", удалив с конца 4 символаs
                service_slug = key[:-4]
                break
        else:
            raise forms.ValidationError('Cannot find service slug in role data')

        try:
            return Service.objects.get(slug=service_slug)
        except Service.DoesNotExist:
            raise forms.ValidationError('Unknown service {0} for role {1}'.format(service_slug, role_data.get('role')))

    def clean_fields(self):
        data_fields = self.cleaned_data['fields']

        if not data_fields:
            return {}

        resource = data_fields.get('resource', '')
        if isinstance(resource, str):
            resource = resource.strip()
            if not resource.isdigit():
                raise forms.ValidationError('Resource should be an integer')

        data_fields['resource'] = None
        if resource:
            try:
                data_fields['resource'] = Resource.objects.get(pk=resource)
            except Resource.DoesNotExist:
                raise forms.ValidationError('Cannot find resource with pk "%s"' % data_fields['resource'])

        return data_fields


class AddUserRoleForm(AddRoleForm):
    login = UserField(label='Сотрудник')


class AddGroupRoleForm(AddRoleForm):
    group = GroupField(label='Группа')


class RemoveUserRoleForm(AddRoleForm):
    login = UserField(label='Сотрудник')
    is_fired = NoValidationField(label='Уволен')


class RemoveGroupRoleForm(AddRoleForm):
    group = GroupField(label='Группа')
    is_deleted = NoValidationField(label='Удалена')


class PlannerGetRolesForm(GetRolesForm):
    TYPES = ('departments', 'members')
    type = forms.ChoiceField(choices=list(zip(TYPES, TYPES)), required=False)
