import inspect
import logging
from functools import wraps

from django.conf import settings
from django.contrib.auth import get_user_model
from django.db.models import Q

from django_idm_api.hooks import RoleStream

from plan.idm import nodes
from plan.idm.adapters import RoleManager
from plan.idm.manager import idm_manager
from plan.internal_roles.models import InternalRole
from plan.roles import models as roles_models
from plan.services import models as services_models
from plan.services.constants.permissions import SERVICE_TAG_APP, SERVICE_TAG_MODEL, CHANGE_SERVICE_TAG_PREFIX
from plan.staff.models import Department, Staff

log = logging.getLogger(__name__)


class ServicePathMixin(object):
    def get_slug_path(self, service__path):
        # дописывает /slug_key/ между slug сервисов
        # /type/services/services_key/meta_infra/meta_infra_key/testnd/testnd_key/*/
        slugs = service__path.split('/')[1:-1]
        slug_path = ''.join('%s/%s_key/' % (slug, slug) for slug in slugs)
        return slug_path

    def get_role_path(self, service__path, role_id):
        return '/type/services/services_key/' + \
               self.get_slug_path(service__path) + \
               '*/role/%d/' % role_id


class InternalRolesBaseStream(RoleStream):

    ACCESS_ROLES = [slug for slug, _ in settings.ABC_INTERNAL_ACCESS_ROLES]

    def values_list(self, queryset):
        return queryset.values_list('pk', 'staff__login', 'role')

    def row_as_dict(self, row):
        pk, staff__login, role = row

        return {
            'login': staff__login,
            'path': '/type/internal/internal_key/%s/' % role,
            '_pk': pk,
        }


class InternalRolesStream(InternalRolesBaseStream):
    name = 'internal'
    next_stream = 'departments'

    def get_queryset(self):
        return (
            InternalRole.objects.exclude(role__in=self.ACCESS_ROLES)
            .select_related('staff')
            .order_by('pk')
        )


class InternalAccessRolesStream(InternalRolesBaseStream):
    name = 'internal'
    next_stream = 'service_tags'

    def get_queryset(self):
        return (
            InternalRole.objects.filter(role__in=self.ACCESS_ROLES)
            .select_related('staff')
            .order_by('pk')
        )


class ServiceTagPermissionsStream(RoleStream):
    name = 'service_tags'

    def values_list(self, queryset):
        return queryset.values_list('pk', 'user__staff__login', 'permission__codename')

    def get_queryset(self):
        return (
            get_user_model().user_permissions.through.objects.filter(
                permission__content_type__app_label=SERVICE_TAG_APP,
                permission__content_type__model=SERVICE_TAG_MODEL,
                permission__codename__startswith=CHANGE_SERVICE_TAG_PREFIX,
            ).select_related('user__staff', 'permission').order_by('pk')
        )

    def row_as_dict(self, row):
        pk, staff_login, permission_codename = row
        return {
            'login': staff_login,
            'path': '/type/service_tags/service_tags_key/%s/' % permission_codename[len(CHANGE_SERVICE_TAG_PREFIX):],
            '_pk': pk,
        }


class DepartmentsStream(RoleStream, ServicePathMixin):
    name = 'departments'
    next_stream = 'members'

    def get_queryset(self):
        return (
            services_models.ServiceMemberDepartment.objects
            .exclude(Q(role=None) | Q(service__state=services_models.Service.states.DELETED))
            .select_related('department', 'department__group', 'service')
            .order_by('pk')
        )

    def values_list(self, queryset):
        return queryset.values_list('pk', 'department__staff_id', 'service__path', 'role_id', 'resource_id')

    def row_as_dict(self, row):
        pk, department__staff_id, service__path, role_id, resource_id = row
        struct = {
            'group': department__staff_id,
            'path': self.get_role_path(service__path, role_id),
            '_pk': pk,
        }

        if resource_id:
            struct['fields'] = {'resource': resource_id}

        return struct


class MembersStream(RoleStream, ServicePathMixin):
    name = 'members'

    def get_queryset(self):
        return (
            services_models.ServiceMember.objects
            .filter(from_department=None)
            .exclude(service__state=services_models.Service.states.DELETED)
            .select_related('staff', 'service')
            .order_by('pk')
        )

    def values_list(self, queryset):
        return queryset.values_list('pk', 'staff__login', 'service__path', 'role_id', 'resource_id')

    def row_as_dict(self, row):
        pk, staff__login, service__path, role_id, resource_id = row
        struct = {
            'login': staff__login,
            'path': self.get_role_path(service__path, role_id),
            '_pk': pk,
        }

        if resource_id:
            struct['fields'] = {'resource': resource_id}

        return struct


def get_membership_by_role(idm_role):
    """
    :param idm_role: словарь, возращаемый ручкой /roles/<id>/
    :return: ServiceMember or ServiceMemberDepartment object
    """

    role = get_role(idm_role)
    service = get_service_by_role(idm_role)

    if idm_role['user']:
        staff = get_staff_by_role(idm_role)

        try:
            return (
                services_models.ServiceMember.objects
                .filter(staff=staff, role=role, service=service)
                .first()
            )
        except IndexError:
            log.warning('Cannot find ServiceMember for idm role %s' % idm_role['id'])

    else:
        department = get_department_by_role(idm_role)

        try:
            return (
                services_models.ServiceMemberDepartment.objects
                .filter(department=department, role=role, service=service)
                .first()
            )
        except IndexError:
            log.warning('Cannot find ServiceMemberDepartment for idm role %s' % idm_role['id'])


def get_department_by_role(idm_role):
    return Department.objects.get(staff_id=idm_role['group']['id'])


def get_staff_by_role(idm_role):
    return Staff.objects.get(login=idm_role['user']['username'])


def get_role(idm_role):
    return roles_models.Role.objects.get(pk=idm_role['node']['data']['role'])


def get_role_by_membership(service_member):
    manager = idm_manager()

    params = {
        'system': settings.ABC_IDM_SYSTEM_SLUG,
        'path': nodes.get_role_node(service_member.service, service_member.role).value_path,
        'type': 'active',
        '_requester': settings.ABC_ZOMBIK_LOGIN,
    }

    if isinstance(service_member, services_models.ServiceMember):
        params['user'] = service_member.staff.login

    elif isinstance(service_member, services_models.ServiceMemberDepartment):
        params['group'] = service_member.department.group_id

    else:
        raise ValueError('Bad member type: %s' % type(service_member))

    roles = RoleManager.get_roles(manager, params)

    if not roles['objects']:
        log.warning('Cannot found role by member: %s', service_member)
        return

    return roles['objects'][0]


def get_service_by_role(idm_role):
    slug = get_service_slug(idm_role['node']['data'])
    service = services_models.Service.objects.get(slug=slug)
    return service


def get_service_slug(role_data):
    """
        Находит в словаре роли слаг сервиса и отрезает "_key"
    """
    rev_data = {v: k for k, v in role_data.items()}
    return rev_data['*'][:-4]


def crowdtest_disabled(function):
    @wraps(function)
    def wrapper(*args, **kwargs):
        signature = inspect.signature(function)
        if signature.return_annotation and signature.return_annotation is not inspect.Signature.empty:
            return signature.return_annotation()

    return wrapper
