import logging
import random

from django.conf import settings
from django.db import transaction
from django.db.models import Q
from django_idm_api.exceptions import BadRequest
from django_idm_api.hooks import BaseHooks

import waffle

from plan.api.idm import forms
from plan.api.idm import helpers
from plan.internal_roles.models import InternalRole
from plan.internal_roles.utils import(
    add_perm,
    assign_perms_for_internal_role,
    remove_perm,
    remove_perms_for_internal_role,
)
from plan.roles.models import Role
from plan.roles.tasks import deprive_role
from plan.services import tasks
from plan.services.models import ServiceMember, ServiceMemberDepartment, Service


log = logging.getLogger(__name__)


class Hooks(BaseHooks):

    GET_ROLES_STREAMS = [helpers.InternalRolesStream, helpers.DepartmentsStream, helpers.MembersStream]
    INTERNAL_ROLES_SOURCE = settings.ABC_INTERNAL_ROLES

    @transaction.atomic
    def add_role_impl(self, login, role_data, fields, **kwargs):
        log.info('Attempt to add user role from IDM: login %s, role_data %r, fields %r, kwargs %r',
                 login, role_data, fields, kwargs)

        form = forms.AddUserRoleForm({'login': login, 'role_data': role_data, 'fields': fields})
        if not form.is_valid():
            raise BadRequest(form.get_error_message())

        data = form.cleaned_data

        person = data['login']
        role = data['role']

        if role_data['type'] == 'internal':
            if role == 'superuser':
                person.user.is_superuser = True  # для админки
                person.user.is_staff = True
                person.user.save()

            if role == 'support':
                person.user.is_staff = True
                person.user.save()

            InternalRole.objects.get_or_create(
                role=role,
                staff=person,
            )

            assign_perms_for_internal_role(role, person)

        elif role_data['type'] == 'service_tags':
            add_perm(role, person)

        else:
            service = data['service']
            resource = data['fields'].get('resource')
            activate = False

            membership, created = ServiceMember.all_states.get_or_create(
                service=service,
                role=role,
                staff=person,
                resource=resource,
                from_department=None,
            )
            if membership.state != ServiceMember.states.ACTIVE:
                membership.activate()
                activate = True

            if not created and not activate:
                return {'code': 1, 'warning': 'User already has this role in this service.'}

            log.info('New personal membership created: %s', membership)

            # выдача роли руководителя требует дополнительных действий
            if role.code == Role.EXCLUSIVE_OWNER:
                service.owner = person
                service.save(update_fields=['owner'])

                log.info('Owner of %s changed to %s', service, person.login)

                # если зама сделали руководителем, у него надо отобрать роль зама
                try:
                    deputy = ServiceMember.objects.get(
                        service=service,
                        staff=person,
                        role__code=Role.DEPUTY_OWNER
                    )
                    deprive_role.apply_async_on_commit(
                        args=[
                            deputy.id,
                            'Роль зама отозвана, так как пользователь стал руководителем сервиса.'
                        ],
                        countdown=settings.ABC_DEFAULT_COUNTDOWN
                    )
                except ServiceMember.DoesNotExist:
                    pass

                tasks.drop_chown_requests.delay_on_commit(data['service'].pk, data['login'].login)

            if data['role'].code in Role.RESPONSIBLES:
                # Пhосим IDM пересчитать подтверждающих у запрошенных ролей

                from plan.services.tasks import rerequest_roles
                rerequest_roles.apply_async_on_commit(
                    args=[data['service'].pk],
                    countdown=settings.ABC_DEFAULT_COUNTDOWN
                )

            tasks.notify_staff.delay_on_commit(data['service'].id)

            if resource:
                return {'data': {'resource': resource.pk}}

    @transaction.atomic
    def add_group_role_impl(self, group, role_data, fields, **kwargs):
        """Выдаёт группе с ID group роль role с дополнительными полями fields"""
        log.info('Attempt to add group role from IDM: group %s, role_data %r, fields %r, kwargs %r',
                 group, role_data, fields, kwargs)

        form = forms.AddGroupRoleForm({'group': group, 'role_data': role_data, 'fields': fields})
        if not form.is_valid():
            raise BadRequest(form.get_error_message())

        data = form.cleaned_data

        service = data['service']

        department = data['group']
        activate = False

        resource = data['fields']['resource'] if data['fields'].get('resource') else None
        department_member, created = ServiceMemberDepartment.all_states.get_or_create(
            service=service,
            role=data['role'],
            department=department,
            resource=resource,
        )

        if department_member.state != ServiceMemberDepartment.states.ACTIVE:
            department_member.activate()
            activate = True

        if not created and not activate:
            return {'code': 1, 'warning': 'Department already has this role in this service.'}

        log.info('New group membership created: %s', department_member.pk)

        tasks.update_service_department_members.apply_async_on_commit(
            args=[service.id],
            countdown=settings.ABC_DEFAULT_COUNTDOWN + random.randint(0, 20)
        )

        if resource:
            return {'data': {'resource': resource.pk}}

    @transaction.atomic
    def remove_role_impl(self, login, role_data, fields, is_fired, **kwargs):
        log.info('Attempt to remove role from IDM: login %s, role_data %r, fields %r, is_fired %s, kwargs %r',
                 login, role_data, fields, is_fired, kwargs)

        form = forms.RemoveUserRoleForm({
            'login': login,
            'role_data': role_data,
            'fields': fields,
            'is_fired': is_fired,
        })
        if not form.is_valid():
            raise BadRequest(form.get_error_message())

        data = form.cleaned_data

        if data['role_data']['type'] == 'internal':
            if data['role'] == 'superuser':
                data['login'].user.is_superuser = False  # для админки
                data['login'].user.is_staff = False
                data['login'].user.save()

            if data['role'] == 'support':
                data['login'].user.is_staff = False
                data['login'].user.save()

            InternalRole.objects.filter(
                role=data['role'],
                staff=data['login'],
            ).delete()

            remove_perms_for_internal_role(data['role'], data['login'])

        elif data['role_data']['type'] == 'service_tags':
            remove_perm(data['role'], data['login'])

        else:
            service = data['service']
            role = data['role']
            try:
                membership = ServiceMember.objects.get(
                    service=service,
                    role=role,
                    staff=data['login'],
                    resource=data['fields']['resource'] if data['fields'].get('resource') else None,
                    from_department=None,  # IDM при aware присылает только персональные роли
                )
            except ServiceMember.DoesNotExist:
                warning_message = 'Try to remove nonexistent role: service %s, staff %s, role %s' % (
                    service.slug, data['login'].login, role
                )
                log.warning(warning_message)

                if waffle.switch_is_active('idm_api_bad_request'):
                    raise BadRequest(warning_message)
                else:
                    return

            log.info('Deprive role %s for %s', membership.role, membership.staff)
            membership.deprive()

            if role.code == Role.EXCLUSIVE_OWNER and service.is_alive:
                # Сюда мы можем попасть в двух ситуациях
                # 1. назначили нового руководителя, старого отозвали - ничего делать не надо
                # 2. роль руководителя отозвали в IDM или он уволился – чистим у сервиса owner

                heads = service.members.owners()
                if not heads:
                    service.owner = None
                    service.save(update_fields=('owner',))

            tasks.notify_staff.delay_on_commit(service.id)

    @transaction.atomic
    def remove_group_role_impl(self, group, role_data, fields, is_deleted, **kwargs):
        """Отбирает у пользователя login роль role с дополнительными данными data
        и флагом уволенности (is_fired)
            Может бросить UserNotFound и RoleNotFound
        """
        log.info('Attempt to remove group role from IDM: group %s, role_data %r, fields %r, is_deleted %s, kwargs %r',
                 group, role_data, fields, is_deleted, kwargs)

        form = forms.RemoveGroupRoleForm({
            'group': group,
            'role_data': role_data,
            'fields': fields,
            'is_deleted': is_deleted,
        })
        if not form.is_valid():
            raise BadRequest(form.get_error_message())

        data = form.cleaned_data
        department = data['group']

        try:
            department_member = ServiceMemberDepartment.objects.get(
                service=data['service'],
                role=data['role'],
                department=department,
                resource=data['fields']['resource'] if data['fields'].get('resource') else None,
            )
        except ServiceMemberDepartment.DoesNotExist:
            warning_message = 'Try to remove nonexistent role: service %s, group %s, role %s' % (
                data['service'].slug, data['group'], data['role']
            )
            log.warning(warning_message)

            if waffle.switch_is_active('idm_api_bad_request'):
                raise BadRequest(warning_message)
            else:
                return

        department_member.deprive()

        log.info('Membership %s removed', department_member)
        tasks.notify_staff.delay_on_commit(data['service'].id)

    def info(self):
        return {
            'code': 0,
            'roles': {
                'slug': 'type',
                'name': {
                    'ru': 'Типы ролей',
                    'en': 'Role types',
                },
                'values': {
                    'services': {
                        'name': {
                            'ru': 'Роли в сервисах',
                            'en': 'Roles in services',
                        },
                        'roles': {
                            'slug': 'services_key',
                            'name': {
                                'ru': 'Сервисы',
                                'en': 'Services',
                            },
                            'values': self.nodes_by_services(),
                        },
                        'fields': [
                            {
                                'slug': 'resource',
                                'name': {
                                    'ru': 'Идентификатор ресурса',
                                    'en': 'Resource id',
                                },
                                'type': 'charfield',
                                'required': False,
                                'options': {
                                    'blank_allowed': False,
                                }
                            }
                        ]
                    },
                    'internal': {
                        'name': {
                            'ru': 'Роли ABC',
                            'en': 'ABC internal roles',
                        },
                        'roles': {
                            'slug': 'internal_key',
                            'name': {
                                'ru': 'Роли',
                                'en': 'Roles',
                            },
                            'values': self.internal_roles(),
                        }
                    }
                }
            }
        }

    def internal_roles(self):
        tree = {}
        for role in self.INTERNAL_ROLES_SOURCE:
            tree[role[0]] = role[1]

        return tree

    def nodes_by_services(self):
        services = list(Service.objects.alive())
        services_by_parent = {}
        tree = {}
        for service in services:
            services_by_parent.setdefault(service.parent_id, []).append(service)

        for service in services_by_parent[None]:
            tree.update(self.make_service_struct(service, services_by_parent))

        return tree

    def make_service_struct(self, service, services_by_parent):
        data = {
            service.slug: {
                'name': {
                    'ru': (service.name
                           if service.state in Service.states.ACTIVE_STATES
                           else service.name + ' (Закрыт)'),
                    'en': (service.name_en
                           if service.state in Service.states.ACTIVE_STATES
                           else service.name_en + ' (Closed)'),
                },
                'unique_id': 'service_{0}'.format(service.id),
                'is_exclusive': False,
                'roles': {
                    'values': {
                        '*': {
                            'name': {
                                'ru': 'Роли сервиса',
                                'en': 'Service roles',
                            },
                            'unique_id': 'service_{0}_roles'.format(service.id),
                            'is_exclusive': False,
                            'roles': {
                                'values': {},
                                'name': {
                                    'ru': 'Роли сервиса {0}'.format(service.name),
                                    'en': 'Service {0} roles'.format(service.name_en),
                                },
                                'slug': 'role',
                            },
                        },
                    },
                    'slug': '{0}_key'.format(service.slug),
                    'name': {
                        'ru': 'Подсервисы и роли',
                        'en': 'Subservices and roles',
                    },
                },
            },
        }

        for role in Role.objects.filter(Q(service=None) | Q(service=service)).order_by('pk'):
            data[service.slug]['roles']['values']['*']['roles']['values'][str(role.pk)] = {
                'name': {
                    'ru': role.name,
                    'en': role.name_en,
                },
                'unique_id': 'service_{0}_role_{1}'.format(service.id, role.id),
                'is_exclusive': (role.code == Role.EXCLUSIVE_OWNER),
            }
            if role.service_id is None:
                data[service.slug]['roles']['values']['*']['roles']['values'][str(role.pk)]['set'] = str(role.pk)

        for child in services_by_parent.get(service.id, []):
            data[service.slug]['roles']['values'].update(self.make_service_struct(child, services_by_parent))

        return data
