import logging

from rest_framework import viewsets, serializers

from plan.api import base
from plan.api.mixins import DefaultFieldsMixin
from plan.api.serializers import CompactStaffSerializer, LegacyDepartmentSerializer
from plan.staff.models import Department, Staff
from plan.api.filters import CustomModelMultipleChoiceFilter

log = logging.getLogger(__name__)


class PersonSerializer(CompactStaffSerializer):
    department = base.CompactDepartmentSerializer()
    abc_id = serializers.IntegerField(source='id')

    class Meta:
        model = Staff
        fields = [
            'id',
            'login',
            'first_name',
            'last_name',
            'uid',
            'name',
            'department',
            'is_dismissed',
            'abc_id',
        ]


class LegacyPersonSerializer(CompactStaffSerializer):
    id = serializers.IntegerField()
    full_name = serializers.SerializerMethodField()

    class Meta:
        model = Staff
        fields = [
            'id',
            'login',
            'full_name',
            'is_dismissed',
        ]

    def get_full_name(self, obj):
        return obj.get_full_name()


class LegacyPersonSerializerWithDepartment(LegacyPersonSerializer):
    department = LegacyDepartmentSerializer()

    class Meta:
        model = Staff
        fields = LegacyPersonSerializer.Meta.fields + ['department']


class LegacyPersonWithSubordinates(LegacyPersonSerializerWithDepartment):
    has_subordinates = serializers.SerializerMethodField()

    class Meta:
        model = Staff
        fields = LegacyPersonSerializerWithDepartment.Meta.fields + ['has_subordinates']

    def get_has_subordinates(self, obj):
        queryset = obj.get_subordinate_logins(only_direct=False)
        return bool(queryset)


class PersonFilter(base.ServiceDescendantsFilterSet):
    department = CustomModelMultipleChoiceFilter(
        queryset=Department.objects.all(),
        distinct=True,
    )

    class Meta:
        model = Staff
        fields = {
            'login': ['exact', 'in'],
            'uid': ['exact', 'in'],
            'department': ['exact'],
        }


class PersonView(base.OrderingMixin, viewsets.ModelViewSet):
    serializer_class = PersonSerializer
    filter_class = PersonFilter
    http_method_names = ['get']
    _permissions_to_proceed = 'view_hierarchy'
    queryset = Staff.objects.all().select_related('department')


class V4PersonView(DefaultFieldsMixin, PersonView):
    default_fields = ['id', 'login', 'uid']
