import inspect
from typing import Collection

import logging

from django.db import models
from django.utils.functional import cached_property

from rest_framework import relations, serializers
from rest_framework.response import Response

from plan.api.fields import IntegerInterfacedSerializerField

DEFAULT_VIEW_ACTIONS = {'create', 'retrieve', 'update', 'partial_update', 'list'}
logger = logging.getLogger(__name__)


class OrderingMixin(object):
    ordering_fields = ('id',)

    @cached_property
    def ordering(self):
        """
        Если использовать ORDER_BY по двум полям, то постгрес всегда сортирует и получается быстро
        Если не делать этот страный хак, то запрос выполняется на порядок дольше
        """
        ordering = ['id']
        additional_ordering_field = getattr(self, 'additional_ordering_field', None)
        if additional_ordering_field:
            ordering.append(additional_ordering_field)
        return ordering


class NoPaginationListMixin(object):
    """
    Возвращает не просто список объектов, а {"results": []}, чтобы фронтенд не сломался в местах, где ожидает пагинацию
    """
    def list(self, request, *args, **kwargs):
        queryset = self.filter_queryset(self.get_queryset())
        serializer = self.get_serializer(queryset, many=True)
        return Response({'results': serializer.data})


class DontFilterOnGetObjectMixin(object):
    # копия оригинального метода, но без вызова self.filter_queryset
    def get_object(self):
        from rest_framework.generics import get_object_or_404

        queryset = self.get_queryset()

        # Perform the lookup filtering.
        lookup_url_kwarg = self.lookup_url_kwarg or self.lookup_field

        assert lookup_url_kwarg in self.kwargs, (
            'Expected view %s to be called with a URL keyword argument '
            'named "%s". Fix your URL conf, or set the `.lookup_field` '
            'attribute on the view correctly.' %
            (self.__class__.__name__, lookup_url_kwarg)
        )

        filter_kwargs = {self.lookup_field: self.kwargs[lookup_url_kwarg]}
        obj = get_object_or_404(queryset, **filter_kwargs)

        # May raise a permission denied
        self.check_object_permissions(self.request, obj)

        return obj


class SelectiveFieldsMixin(object):
    explicit_fields = ()

    @cached_property
    def fields(self):
        fields = super(SelectiveFieldsMixin, self).fields

        if not hasattr(self, '_context'):
            # если так, то список полей запрашивают до начала обработки запроса
            return fields

        # фильтруем только если мы не nested serializer,
        # или если родитель - сериалайзер с many=True
        is_nested = False
        is_root = self.root == self
        parent_is_list_root = self.parent == self.root and getattr(self.parent, 'many', False)
        if not (is_root or parent_is_list_root):
            is_nested = True

        try:
            request = self.context['request']
        except KeyError:
            return fields

        try:
            filter_fields = request.query_params.get('fields', None).split(',')
        except AttributeError:
            filter_fields = None

        existing = set(fields.keys())

        # если ?fields= не прислали, то не фильтруем поля
        if filter_fields is None:
            allowed = existing - set(self.explicit_fields)

        # парсим фильтры для вложенных сериализаторов
        elif is_nested:

            # строим префикс для запроса полей
            family = []
            member = self
            recursion_count = 0
            while member.parent:
                if recursion_count > 5:
                    # защищаемся от излишней рекурсии и отдаем все поля,
                    # если спустились слишком глубоко
                    return fields

                if member.field_name:
                    family.insert(0, member.field_name)
                member = member.parent
                recursion_count += 1

            if family:
                prefix = '.'.join(family) + '.'
                allowed = set((
                    # тут если в фильтре после префикса детализация
                    # (например resource.type.name),
                    # то выкидываем её - она обработается при следующем вызове
                    # этого метода уже у вложенного сериализатора
                    filter_field[len(prefix):].split('.')[0]
                    for filter_field in filter_fields
                    if filter_field.startswith(prefix)
                ))

                # если фильтры для вложенных не обнаружили, значит показываем всё
                if not allowed:
                    return fields

        else:
            allowed = set((
                # если прислали подзапрос для вложенных подсериализаторов
                # типа subserializer.name, выдергиваем имя
                # подсериализатора, чтоб он дальше отрисовался дальше
                field.strip().split('.')[0]
                for field in filter_fields
                if field
            ))

        # дропаем все поля, которые не запросили
        for field in existing:
            if field not in allowed:
                fields.pop(field, None)

        return fields


class UnwrapPaginatorMixinWithSubquery(object):
    def unwrap(self, queryset, request, view=None):
        q = getattr(queryset, '_true_query', queryset)
        pks = (
            data['id']
            for data
            in super(UnwrapPaginatorMixinWithSubquery, self).paginate_queryset(q.values('id', 'modified_at'), request, view=view)
        )
        clone = queryset._clone()
        clone.query.clear_limits()
        return list(clone.filter(id__in=pks))

    def paginate_queryset(self, queryset, request, view=None):
        if getattr(view, 'use_unwrap', False):
            return self.unwrap(queryset, request, view)
        else:
            return super(UnwrapPaginatorMixinWithSubquery, self).paginate_queryset(queryset, request, view=view)


class FieldsMappingMixin(object):
    def __init__(self, *args, **kwargs):
        queryset = kwargs.pop('queryset', None)
        super(FieldsMappingMixin, self).__init__(*args, **kwargs)
        if self.parent is not None and queryset is None:
            raise ValueError('Serializer field must have queryset')
        self.queryset = queryset
        self.fields_mapping = {}
        for parent in reversed(inspect.getmro(self.__class__)):
            fields = getattr(parent, 'fields_mapping_', None)
            if fields:
                self.fields_mapping.update(fields)

    @staticmethod
    def process_mapping_field(field_name, serializer, only_fields, prefix):
        for field in serializer.fields_mapping[field_name]:
            only_fields.append(f'{prefix}__{field}' if prefix else field)

    @staticmethod
    def process_list_serializer(serializer, model_name, prefix, prefetch):
        if not serializer.fields:
            return
        prefetch.append(serializer.get_prefetch(model_name, prefix))

    @staticmethod
    def process_detail_serializer(serializer, serializers_stack, prefix, select):
        if not serializer.fields:
            return
        serializers_stack.append((serializer, prefix))
        select.append(prefix)

    def process_integer_interfaced_serializer(self, field, model_name, prefix, serializers_stack, prefetch, select):
        serializer = field.child_relation.get_serializer(None)
        if isinstance(field, relations.ManyRelatedField):
            self.process_list_serializer(serializer, model_name, prefix, prefetch)
        else:
            self.process_detail_serializer(serializer, serializers_stack, prefix, select)

    def process_one_serializer(self, serializer, model_name, prefix, serializers_stack, only, select, prefetch):
        only.append(f'{prefix}__id' if prefix else 'id')
        for field_name, value in serializer.fields.items():
            field = field_name if value.source == '*' else value.source
            new_prefix = f'{prefix}__{field}' if prefix else field
            if field_name in serializer.fields_mapping:
                self.process_mapping_field(
                    field_name=field_name,
                    serializer=serializer,
                    only_fields=only,
                    prefix=prefix,
                )
            elif isinstance(value, serializers.ListSerializer):
                self.process_list_serializer(
                    serializer=value.child,
                    model_name=model_name,
                    prefix=new_prefix,
                    prefetch=prefetch,
                )
            elif isinstance(value, serializers.Serializer):
                self.process_detail_serializer(
                    serializer=value,
                    serializers_stack=serializers_stack,
                    prefix=new_prefix,
                    select=select,
                )
            elif isinstance(getattr(value, 'child_relation', None), IntegerInterfacedSerializerField):
                self.process_integer_interfaced_serializer(
                    field=value,
                    model_name=model_name,
                    prefix=new_prefix,
                    serializers_stack=serializers_stack,
                    prefetch=prefetch,
                    select=select,
                )
            else:
                only.append(new_prefix)

    def get_query_fields(self, model_name):
        assert hasattr(self, '_context'), 'Serializer must have _context'
        only_fields = []
        select_related_fields = []
        prefetch_related_fields = []
        serializers_stack = [(self, '')]
        while serializers_stack:
            serializer, db_prefix = serializers_stack.pop()
            self.process_one_serializer(
                serializer,
                model_name,
                db_prefix,
                serializers_stack,
                only_fields,
                select_related_fields,
                prefetch_related_fields,
            )
        return only_fields, select_related_fields, prefetch_related_fields

    def get_prefetch(self, model_name, lookup):
        only_fields, select_fields, prefetch_fields = self.get_query_fields(self.queryset.model._meta.model_name)
        additional_field = f'{model_name}_id'

        # Для связи ManyToOne нам нужно выбрать id объекта, с которым будем делать prefetch,
        # но часто такого поля нет в сериализаторе.
        if hasattr(self.queryset.model, additional_field):
            only_fields.append(additional_field)
        queryset = self.queryset.only(*only_fields).select_related(*select_fields).prefetch_related(*prefetch_fields)
        return models.Prefetch(lookup, queryset=queryset)


class FieldsByPermissionsCutterMixin:
    def dispatch(self, request, *args, **kwargs):
        if request.method == 'GET':  # other methods are blocked by CanEditMiddleWare
            allowed_fields = self.allowed_fields(request.user)
            if 'fields' not in request.GET:
                filtered_fields = allowed_fields
            else:
                filtered_fields = self.bad_fields(request.GET['fields'], allowed_fields)
            filtered_params = self.bad_params(request.GET, allowed_fields)

            old_value = request.GET._mutable
            request.GET._mutable = True
            request.GET['fields'] = ','.join(filtered_fields)
            for param in filtered_params:
                del request.GET[param]
            request.GET._mutable = old_value

        return super(FieldsByPermissionsCutterMixin, self).dispatch(request, *args, **kwargs)

    def allowed_fields(self, user) -> Collection[str]:
        all_perms = user.get_all_permissions()
        internal_perms = [
            item.split('.')[-1] for item in all_perms if item.startswith('internal_roles')
        ]
        empty = set()
        fields = set()
        for perm in internal_perms:
            fields.update(self.PERMISSION_CODENAME_MAPPER.get(perm, empty))
        fields.discard(self._EMPTY)
        return fields

    def bad_params(self, params: dict, allowed_fields: Collection[str]) -> Collection[str]:
        all_fields = {f for p in self.PERMISSION_CODENAME_MAPPER.values() for f in p}
        forbidden_fields = all_fields - set(allowed_fields)
        forbidden_params = {p for p in params if p.split('__')[0] in forbidden_fields}
        return forbidden_params

    def bad_fields(self, fields: Collection[str], allowed_fields: Collection[str]) -> Collection[str]:
        requested_fields = set(fields.split(','))
        filtered_fields = set()
        for field in requested_fields:
            short_field = field.split('.')[0]
            if short_field in allowed_fields:
                filtered_fields.add(field)
        return filtered_fields


class DefaultFieldsMixin:
    default_fields = None

    def check_fields(self, request):
        if request.method != 'GET' or 'fields' in request.GET:
            return
        old_value = request.GET._mutable
        request.GET._mutable = True
        request.GET['fields'] = ','.join(self.default_fields)
        request.GET._mutable = old_value

    def dispatch(self, request, *args, **kwargs):
        self.check_fields(request)
        return super(DefaultFieldsMixin, self).dispatch(request, *args, **kwargs)


class SelectOnlyFieldsMixin(object):
    default_only_fields = tuple()
    default_select_fields = tuple()
    default_prefetch_fields = tuple()

    def get_queryset(self):
        self.only_fields = None
        if self.action not in DEFAULT_VIEW_ACTIONS:
            return self.queryset
        queryset = self.queryset
        serializer = self.get_serializer_class()(context={'request': self.request})
        only_fields, select_fields, prefetch_fields = serializer.get_query_fields(self.queryset.model._meta.model_name)
        # ToDo: убрать, кога везде начнем использовать value
        if only_fields:
            only_fields.extend(self.default_only_fields)
            self.only_fields = only_fields
            queryset = queryset.only(*self.only_fields)
        if select_fields:
            select_fields.extend(self.default_select_fields)
            queryset = queryset.select_related(*select_fields)
        if prefetch_fields:
            prefetch_fields.extend(self.default_prefetch_fields)
            queryset = queryset.prefetch_related(*prefetch_fields)
        return queryset


class TvmAccessMixin(object):

    TVM_ALLOWED_METHODS = {'GET'}
