from rest_framework import serializers
from django.db import models

from plan.api.fields import (
    CustomPrimaryKeyRelatedField,
    MappingField,
    CustomDateField,
)
from plan.api.mixins import FieldsMappingMixin, SelectiveFieldsMixin
from plan.contacts.models import ContactType
from plan.roles.models import Role, RoleScope
from plan.services.models import Service
from plan.staff.models import Department, Staff
from plan.resources.models import ResourceType

serializer_field_mapping = (
    serializers.ModelSerializer.serializer_field_mapping.copy()
)
serializer_field_mapping[models.DateTimeField] = CustomDateField

INTERNAL_ATTRIBUTES_KEY = 'internal_attributes'


class AttributesSerializer(object):
    @staticmethod
    def get_attributes(obj):
        try:
            attrs = obj['attributes']
        except TypeError:
            attrs = obj.attributes
        result = []
        for key, value in attrs.items():
            if key == INTERNAL_ATTRIBUTES_KEY:
                continue
            if isinstance(value, dict):
                value['name'] = key
                result.append(value)
            else:
                result.append({'name': key, 'value': value})

        return result


class BasePlanModelSerializer(serializers.ModelSerializer):
    serializer_field_mapping = serializer_field_mapping


class ModelSerializer(FieldsMappingMixin, SelectiveFieldsMixin, BasePlanModelSerializer):
    pass


class CompactDepartmentSerializer(ModelSerializer):
    name = MappingField({'ru': 'name', 'en': 'name_en'})
    fields_mapping_ = {'name': ('name', 'name_en')}

    class Meta:
        model = Department
        fields = ['id', 'url', 'name']


class LegacyDepartmentSerializer(ModelSerializer):
    name = serializers.CharField(source='i_name')

    class Meta:
        model = Department
        fields = ['id', 'name']


class CompactStaffSerializer(ModelSerializer):
    first_name = MappingField({'ru': 'first_name', 'en': 'first_name_en'})
    last_name = MappingField({'ru': 'last_name', 'en': 'last_name_en'})
    name = serializers.SerializerMethodField()
    id = serializers.IntegerField(source='staff_id')
    abc_id = serializers.IntegerField(source='id')
    fields_mapping_ = {
        'first_name': ('first_name', 'first_name_en'),
        'last_name': ('last_name', 'last_name_en'),
        'name': ('first_name', 'first_name_en', 'last_name', 'last_name_en'),
        'id': ('staff_id',),
        'abc_id': ('id',),
    }

    class Meta:
        model = Staff
        fields = [
            'id',
            'login',
            'name',
            'first_name',
            'last_name',
            'abc_id',
            'uid',
            'department',
            'is_dismissed',
            'is_robot',
            'affiliation',
            'is_frozen',
            'telegram_account',
        ]

    def get_name(self, obj):
        return {
            'ru': f'{obj.first_name} {obj.last_name}'.strip(),
            'en': f'{obj.first_name_en} {obj.last_name_en}'.strip(),
        }


class UpdateStaffSerializer(ModelSerializer):
    class Meta:
        model = Staff
        fields = ('is_frozen', )


class ContactTypeSerializer(ModelSerializer):
    name = MappingField({'ru': 'name', 'en': 'name_en'})
    fields_mapping_ = {'name': ('name', 'name_en')}

    class Meta:
        model = ContactType
        fields = ['id', 'code', 'validator', 'name']


class CompactServiceSerializer(ModelSerializer):
    parent = CustomPrimaryKeyRelatedField(allow_null=True, queryset=Service.objects.all(), required=False)
    name = MappingField({'ru': 'name', 'en': 'name_en'})
    fields_mapping_ = {'name': ('name', 'name_en')}

    class Meta:
        model = Service
        fields = ('id', 'slug', 'name', 'parent')


class CompactRoleScopeSerializer(ModelSerializer):
    name = MappingField({'ru': 'name', 'en': 'name_en'})
    fields_mapping_ = {'name': ('name', 'name_en')}

    class Meta:
        model = RoleScope
        fields = ['id', 'slug', 'name']


class CompactRoleSerializer(ModelSerializer):
    name = MappingField({'ru': 'name', 'en': 'name_en'})

    scope = CompactRoleScopeSerializer(queryset=RoleScope.objects.all())
    service = CompactServiceSerializer(queryset=Service.objects.all())

    fields_mapping_ = {'name': ('name', 'name_en')}

    class Meta:
        model = Role
        fields = ['id', 'name', 'service', 'scope', 'code']


class CompactResourceTypeSerializer(ModelSerializer):
    supplier = CompactServiceSerializer(queryset=Service.objects.all())

    class Meta:
        model = ResourceType
        fields = [
            'id',
            'name',
            'supplier'
        ]


def serializer_wrapped(obj, obj_type, serializer):
    result = serializer(obj).data
    wrapped = {
        'type': obj_type,
        'value': result,
        'entities': []
    }
    return wrapped
