from rest_framework.generics import ListAPIView
from rest_framework.filters import OrderingFilter, SearchFilter

from plan.api.filters import StaffIdIncludeFilter, StaffIdExcludeFilter
from plan.api.suggests.mixins import SuggestMixin
from plan.api.suggests.pagination import SuggestPagination
from plan.api.suggests.persons.serializers import StaffSuggestSerializer, PersonSuggestForm
from plan.services.models import ServiceMember
from plan.staff.models import Staff


class PersonSuggestView(SuggestMixin, ListAPIView):
    pagination_class = SuggestPagination

    filter_backends = (
        SearchFilter,
        StaffIdIncludeFilter,
        StaffIdExcludeFilter,
        OrderingFilter
    )

    search_fields = ('first_name', 'first_name_en', 'last_name', 'last_name_en', 'login',)
    ordering = ('first_name', 'last_name',)

    serializer_class = StaffSuggestSerializer

    def get_queryset(self):
        queryset = Staff.objects.all()
        form = PersonSuggestForm(data=self.request.query_params)
        form.is_valid(raise_exception=True)
        schedule = form.validated_data.get('schedule')
        if schedule:
            service = schedule.service
            role_q = schedule.get_role_q()
            members_pks = ServiceMember.objects.filter(service=service).filter(role_q).values_list('staff', flat=True)
            queryset = queryset.filter(pk__in=members_pks)

        return queryset
