from rest_framework.generics import ListAPIView
from rest_framework.filters import OrderingFilter, SearchFilter

from plan.api.filters import ExcludeFilter, IncludeFilter
from plan.api.suggests.mixins import SuggestMixin
from plan.api.suggests.pagination import SuggestPagination
from plan.api.suggests.roles.serializers import RoleSuggestSerializer
from plan.roles.models import Role


class RoleSuggestView(SuggestMixin, ListAPIView):
    pagination_class = SuggestPagination

    filter_backends = (
        SearchFilter,
        IncludeFilter,
        ExcludeFilter,
        OrderingFilter
    )

    search_fields = ('name', 'name_en')
    ordering = ('scope__name', 'name')

    serializer_class = RoleSuggestSerializer

    def get_queryset(self):
        queryset = Role.objects.filter(service__isnull=True).select_related('scope', 'service')
        service_id = self.request.query_params.get('service', None)
        if service_id is not None:
            try:
                service_id = int(service_id)
            except ValueError:
                pass
            else:
                queryset |= Role.objects.filter(service_id=service_id)

        can_issue_at_duty_time = self.request.query_params.get('scope__can_issue_at_duty_time', None)
        if can_issue_at_duty_time in ('1', 'True', 'true'):
            queryset = queryset.exclude(
                code__in=Role.CAN_NOT_USE_FOR_DUTY,
            ).exclude(
                scope__can_issue_at_duty_time=False,
            )
        return queryset
