from django.conf import settings
from rest_framework import viewsets
from rest_framework.response import Response

from plan.api.exceptions import FailedDependency
from plan.api.mixins import (
    SelectOnlyFieldsMixin,
    TvmAccessMixin,
    OrderingMixin
)
from plan.api.permissions import TvmAuthenticated
from plan.common.utils.http import Session
from plan.common.utils.tvm import get_tvm_ticket
from plan.services.models import Service
from plan.swagger import SwaggerFrontend


class FrontendSuggestServicesView(viewsets.ReadOnlyModelViewSet,
                                  TvmAccessMixin,
                                  SelectOnlyFieldsMixin,
                                  OrderingMixin):
    default_swagger_schema = SwaggerFrontend
    permission_classes = [TvmAuthenticated]

    def list(self, request, *args, **kwargs):
        text = request.GET.get('text')
        # по умолчанию в intrasearch значение 0, прямо пробрасываю его, если не задано иное
        page = request.GET.get('page', 0)
        # по умолчанию в intrasearch значение 5, прямо пробрасываю его, если не задано иное
        per_page = request.GET.get('per_page', 5)

        status_list = []
        for state in Service.states.ACTIVE_STATES:
            status_list.append(f's_status:"{state}"')

        status_query = ' | '.join(status_list)

        # идём в intrasearch
        params = {
            'layers': 'services',
            'version': 2,
            'text': text,
            'services.page': page,
            'services.per_page': per_page,
            'services.query': status_query,
        }

        headers = {
            'Content-type': 'application/json',
            'X-Ya-Service-Ticket': get_tvm_ticket(destination=str(settings.INTRASEARCH_TVM_ID)),
            'X-Ya-User-Ticket': self.request.tvm_user_ticket,
        }

        with Session() as session:
            response = session.get(settings.INTRASEARCH_ENDPOINT, params=params, headers=headers)

        result = response.json()

        if response.status_code == 200:
            result = result['services']

            services_id = [found_service['id'] for found_service in result['result']]
            services_dict = {service.id: service for service in Service.objects.filter(pk__in=services_id)}
            for found_service in result['result']:
                service = services_dict[found_service['id']]

                found_service['is_base_non_leaf'] = service.is_base_non_leaf()
                found_service['name'] = {
                    'ru': service.name,
                    'en': service.name_en,
                }

        elif response.status_code > 499:
            raise FailedDependency(
                message={
                    'ru': f'Intrasearch вернул {response.status_code}.',
                    'en': f'Intrasearch responded with {response.status_code}.',
                }
            )

        return Response(result)
