from typing import Union, Dict

from rest_framework import validators as drf_validators

from plan.api.exceptions import ValidationError


class MutuallyExclusiveFieldsValidator(object):
    def __init__(self, exclusive_fields):
        self.fields = exclusive_fields

    def __call__(self, attrs, *args, **kwargs):
        if len(set(attrs) & set(self.fields)) > 1:
            raise ValidationError(
                detail='Got mutually exclusive fields: {}'.format(self.fields),
                message={
                    'ru': 'В вашем запросе пришли взаимоисключающие поля: {}'.format(self.fields),
                    'en': 'Your request supplied mutually exclusive fields: {}'.format(self.fields)
                }
            )


class UniqueFieldValidator(drf_validators.UniqueValidator):
    def __init__(self, queryset, field=None, message=None, lookup='exact'):
        self.field = field
        super(UniqueFieldValidator, self).__init__(queryset, message, lookup)

    def set_context(self, serializer_field):
        self.field_name = self.field or serializer_field.source_attrs[-1]
        self.instance = getattr(serializer_field.parent, 'instance', None)

    def filter_queryset(self, value, queryset, field_name):
        """
        Filter the queryset to all instances matching the given attribute.
        Взято из исходников drf с заменой self.field -> self.field_name
        иначе происходит фильтрация по name_ru например, а не по name как нужно нам
        """
        filter_kwargs = {'%s__%s' % (self.field_name, self.lookup): value}
        return drf_validators.qs_filter(queryset, **filter_kwargs)


class UniqueTogetherLookupValidator(drf_validators.UniqueTogetherValidator):
    def __init__(self, queryset, fields_with_lookup, message=None):
        """ выкусываем lookups и складываем отдельно """
        self.lookup = {}
        fields = []
        for field in fields_with_lookup:
            if '__' in field:
                field, lookup = field.split('__')
            else:
                lookup = 'exact'
            self.lookup[field] = lookup
            fields.append(field)
        super(UniqueTogetherLookupValidator, self).__init__(queryset, fields, message)

    def enforce_required_fields(self, attrs, serializer):
        """
        Скопировал метод из 3.6 версии, в новой версии он работает с логикой
        отличной от требуемой для MappingField (так как мы передаем в source = '*')
        """
        if serializer.instance is not None:
            return

        missing_items = {
            field_name: self.missing_message
            for field_name in self.fields
            if field_name not in attrs
        }
        if missing_items:
            raise ValidationError(missing_items, code='required')

    def filter_queryset(self, attrs, queryset, serializer):
        """
        Filter the queryset to all instances matching the given attributes.
        """
        # If this is an update, then any unprovided field should
        # have it's value set based on the existing instance attribute.
        if serializer.instance is not None:
            for field_name in self.fields:
                if field_name not in attrs:
                    attrs[field_name] = getattr(self.instance, field_name)

        # Determine the filter keyword arguments and filter the queryset.
        filter_kwargs = {
            f'{field_name}__{self.lookup[field_name]}': attrs[field_name]
            for field_name in self.fields
        }
        return drf_validators.qs_filter(queryset, **filter_kwargs)


class MappingFieldUniqueValidator(drf_validators.UniqueValidator):
    def set_context(self, serializer_field):
        self.field_name = serializer_field.field_name
        self.mapping = serializer_field.mapping
        self.instance = getattr(serializer_field.parent, 'instance', None)

    def __call__(self, value, *args, **kwargs):
        duplicates_found = []
        fields_to_scan = {
            field_name: new_value
            for field_name, new_value in value.items()
            if field_name in self.mapping.values()
        }

        for field_name, new_value in fields_to_scan.items():
            queryset = self.queryset.filter(**{field_name: new_value})
            if self.instance:
                queryset = queryset.exclude(id=self.instance.id)

            if queryset.exists():
                duplicates_found.append(field_name)

        if duplicates_found:
            reverse_map = {value: key for key, value in self.mapping.items()}
            mapped_field_names = ', '.join(
                ['"{}"'.format(reverse_map.get(field_name)) for field_name in duplicates_found]
            )
            raise ValidationError(
                message={
                    'ru': 'Значения ключей {} поля {} должны быть уникальными.'.format(
                        mapped_field_names, self.field_name
                    ),
                    'en': 'Values of keys {} of field {} must be unique.'.format(
                        mapped_field_names, self.field_name
                    ),
                }
            )


class ServiceNameValidator:
    def validate_name(self, value: str):
        if len(value) <= 1:
            raise ValidationError(
                message={
                    'ru': 'Имя сервиса должно иметь длину не менее двух символов',
                    'en': 'Service name must have at least two characters long'
                }
            )
        if not any(char.isalpha() for char in set(value)):
            raise ValidationError(
                message={
                    'ru': 'В имени сервиса должна должна быть минимум одна буква',
                    'en': 'Service name must contain at least one letter'
                }
            )

    def __call__(self, value: Union[str, Dict[str, str]], *args, **kwargs):
        if isinstance(value, dict):
            for name in value.values():
                self.validate_name(name)
        else:
            self.validate_name(value)
