import json
import logging
import collections

from waffle.models import Switch
from rest_framework import viewsets

from django.conf import settings
from django.middleware.csrf import rotate_token
from django.views.decorators.csrf import ensure_csrf_cookie
from rest_framework.response import Response
from rest_framework.views import APIView
from rest_framework import serializers

from plan.api.exceptions import Unauthorized
from plan.api.intranet.persons import PersonSerializer
from plan.api.mixins import DefaultFieldsMixin
from plan.api.serializers import BasePlanModelSerializer
from plan.common.internal_roles import get_internal_roles
from plan.swagger import SwaggerFrontend

NOTIFIER_NAME = 'notice_'
logger = logging.getLogger(__name__)

NOTICE_ERROR = 'error'
NOTICE_INFO = 'info'
NOTICE_WARNING = 'warning'
NOTICE_LEVEL = {
    NOTICE_ERROR: 1,
    NOTICE_WARNING: 2,
    NOTICE_INFO: 3,
}
DEFAULT_SUBTYPE = 'common'


class UserView(APIView):
    def get(self, request, *args, **kwargs):
        person = request.person.staff
        serializer = PersonSerializer(person)
        if serializer.is_valid:
            return Response(serializer.data)
        else:
            raise Unauthorized()


class CSRFView(APIView):
    authentication_classes = ()

    @staticmethod
    @ensure_csrf_cookie
    def get(request):
        rotate_token(request)
        return Response({'csrf_token': request.META['CSRF_COOKIE']})


class NotifierSerializer(BasePlanModelSerializer):
    type = serializers.SerializerMethodField()
    message = serializers.SerializerMethodField()
    expiration_time = serializers.SerializerMethodField()
    cta = serializers.SerializerMethodField()
    link = serializers.SerializerMethodField()

    class Meta:
        model = Switch
        fields = ('id', 'name', 'type', 'message', 'expiration_time', 'cta', 'link')

    def get_type(self, notice):
        return notice['note']['type']

    def get_message(self, notice):
        return notice['note']['message']

    def get_expiration_time(self, notice):
        return notice['note']['expiration_time']

    def get_cta(self, notice):
        if 'cta' in notice['note']:
            return notice['note']['cta']
        return

    def get_link(self, notice):
        if 'link' in notice['note']:
            return notice['note']['link']
        return


class NotifierView(viewsets.ModelViewSet):
    """
    Отдаём информацию для плашки уведомлений
    """
    serializer_class = NotifierSerializer
    queryset = Switch.objects.all()
    http_method_names = ['get']

    _permissions_to_proceed = 'can_view'

    def get_queryset(self):
        subtypes = self.request.query_params.get('subtype', DEFAULT_SUBTYPE).split(',')

        notices = collections.defaultdict(list)

        for subtype, notice in self.valid_notices(subtypes=subtypes):
            if self.is_permitted(notice):
                notices[subtype].append(notice)

        result = []
        for _, subtype_group in sorted(notices.items(), key=lambda x: x[0]):
            notice = min(
                subtype_group,
                key=lambda n: (NOTICE_LEVEL[n['note']['type']], n['name']),
                default=None
            )
            if notice:
                result.append(notice)

        return result

    def valid_notices(self, subtypes):
        switch_objs = Switch.objects.filter(
            active=True,
            name__startswith=NOTIFIER_NAME,
        ).exclude(
            name__in=self.request.COOKIES,
        )
        for instance in switch_objs:
            try:
                note = json.loads(instance.note)
            except ValueError:
                logger.exception('WaffleNotifier: Invalid JSON in waffle notifier %s', instance.name)
                continue
            notice = {
                'id': instance.id,
                'name': instance.name,
                'note': note,
            }
            name = instance.name

            type_ = note.get('type')
            if not (type_ and type_ in NOTICE_LEVEL):
                logger.exception('WaffleNotifier: unknown type in notice {}'.format(name))
                continue

            subtype = note.get('subtype', DEFAULT_SUBTYPE)
            if subtypes and subtype not in subtypes:
                continue

            message = note.get('message')
            if not message:
                logger.exception('WaffleNotifier: no message in notice {}'.format(name))
                continue

            expiration_time = note.get('expiration_time')
            if not expiration_time:
                logger.exception('WaffleNotifier: no expiration_time in notice {}'.format(name))
                continue

            yield subtype, notice

    def is_permitted(self, notice):
        permissions = set(notice['note'].get('permissions', {'can_view'}))
        return permissions.issubset(get_internal_roles(self.request.user))


class V4NotifierView(DefaultFieldsMixin, NotifierView):
    default_swagger_schema = SwaggerFrontend
    default_fields = ['id', 'name', 'type']


class FrontendPermissionsView(viewsets.ViewSet):
    default_swagger_schema = SwaggerFrontend

    http_method_names = ['get']
    EXPECTED_PERMISSION_SET = (
        set(settings.ABC_INTERNAL_ROLES_PERMISSIONS.get('full_access', set()))
    )
    _permissions_to_proceed = 'can_view'

    def list(self, request, pk=None):
        internal_perms = get_internal_roles(request.user)
        permissions_for_front = internal_perms & self.EXPECTED_PERMISSION_SET
        perm_list = list(permissions_for_front)
        return Response(data={'results': perm_list}, content_type='application/json')
