import logging

from plan.services.api.departments import (
    DepartmentsView,
    V4DepartmentsView,
    FrontendDepartmentsView,
)
from plan.services.api.members import MembersView, V3MembersView, V4MembersView
from plan.services.models import ServiceMember, ServiceMemberDepartment
from plan.services.state import SERVICEMEMBER_STATE
from plan.services.api.frontend import FrontendMembersView


log = logging.getLogger(__name__)


class PermissiveMembersMixin(object):
    """
    Создание и удаление ролей производится без согласования c IDM.

    Может использоваться в изолированных окружениях без доступа к IDM. В том
    числе для ассессорских стендов.
    """

    def _request_membership(self, requester, data):
        return {'id': -1}

    def _create_membership(self, data, idm_role_id, expires_at):
        membership, created = ServiceMember.all_states.get_or_create(
            service=data['service'], staff=data['staff'], role=data['role'],
            from_department=None, resource=None,
        )
        if membership.state != SERVICEMEMBER_STATE.ACTIVE:
            membership.activate()
        log.info(f'New personal membership created: {membership}')

    def _deprive_role(self, requester, obj):
        pass

    def _deprive_membership(self, obj):
        obj.deprive()
        log.info(f'Membership of {obj.staff} in {obj.service} deprived')


class PermissiveMembersView(PermissiveMembersMixin, MembersView):
    pass


class PermissiveV3MembersView(PermissiveMembersMixin, V3MembersView):
    pass


class PermissiveV4MembersView(PermissiveMembersMixin, V4MembersView):
    pass


class PermissiveFrontendMembersView(PermissiveMembersMixin, FrontendMembersView):
    pass


class PermissiveDepartmentsMixin(object):
    """
    Создание и удаление ролей производится без согласования c IDM.

    Может использоваться в изолированных окружениях без доступа к IDM. В том
    числе для ассессорских стендов.
    """

    def _request_membership(self, requester, data):
        return {'id': -1}

    def _create_membership(self, data, idm_role_id, expires_at):
        service = data['service']
        department = data['department']
        role = data['role']
        member_department, _ = ServiceMemberDepartment.all_states.get_or_create(
            service=service, role=role, department=department,
        )
        member_department.activate()
        log.info(f'New group membership activated: {member_department.pk}')
        expires_at = None  # на ассесорском стенде функционал не поддержан
        self._create_members(member_department, service, department, role, expires_at=expires_at)
        members = member_department.members(manager='all_states').filter(state=SERVICEMEMBER_STATE.REQUESTED)
        members.activate()

    def _deprive_role(self, requester, obj):
        pass

    def _deprive_membership(self, obj):
        obj.members.deprive()
        obj.deprive()
        log.info(f'Membership of department {obj} in {obj.service} deprived')


class PermissiveDepartmentsView(PermissiveDepartmentsMixin, DepartmentsView):
    pass


class PermissiveV4DepartmentsView(PermissiveDepartmentsMixin, V4DepartmentsView):
    pass


class PermissiveFrontendDepartmentsView(PermissiveDepartmentsMixin, FrontendDepartmentsView):
    pass
