from typing import Iterable, List, Dict

from django.conf import settings

from plan.common.person import Person
from plan.idm.adapters import ApproveRequestManager
from plan.idm.manager import idm_manager
from plan.resources.models import ServiceResource
from plan.roles.models import Role
from plan.services.models import Service
from plan.staff.models import Department, Staff
from plan.suspicion.models import ServiceAppealIssue


def get_approvable_appeals(person):
    appeals = ServiceAppealIssue.objects.for_services_of_person(person).active()
    return appeals


def get_approvable_role_requests(person: Person, services: Iterable[Service]) -> List[Dict]:
    """
    Ходим в IDM и ищем все запросы ролей, которые может заапрувить человек

    services - QuerySet сервисов, которые нас будут интересовать.
        Т.е. поиск сервисов с/без учёта дочерних выносится за пределы метода.
    """

    manager = idm_manager(
        timeout=settings.IDM_GET_APPROVEREQUESTS_TIMEOUT,
        retry=settings.ABC_IDM_FROM_API_RETRY,
    )
    idm_role_requests = ApproveRequestManager.get_list(manager, person.login)

    staff_logins = set()
    service_slugs = set()
    department_staff_ids = set()
    role_ids = set()
    for role_request in idm_role_requests:
        service_slugs.add(role_request['service'])
        staff_logins.add(role_request['requester'])
        role_ids.add(role_request['role'])
        if role_request['user']:
            staff_logins.add(role_request['user'])
        if role_request['group']:
            department_staff_ids.add(role_request['group'])

    services_map = {
        s.slug: s for s in services
        if s.slug in service_slugs
    }

    staff_map = {s.login: s for s in Staff.objects.filter(login__in=staff_logins)}
    departments_map = {d.staff_id: d for d in Department.objects.filter(staff_id__in=department_staff_ids)}
    roles_map = {r.id: r for r in Role.objects.filter(id__in=role_ids).select_related('scope')}

    role_requests = []
    for role_request in idm_role_requests:
        user = None
        if role_request['user']:
            user = staff_map.get(role_request['user'])
            if not user:
                continue

        group = None
        if role_request['group']:
            group = departments_map.get(int(role_request['group']))
            if not group:
                continue

        requester = staff_map.get(role_request['requester'])
        service = services_map.get(role_request['service'])
        if not requester or not service:
            continue

        role_requests.append({
            'idm_request_id': role_request['idm_request_id'],
            'user': user,
            'group': group,
            'requester': requester,
            'role': roles_map.get(int(role_request['role'])),
            'service': service
        })

    return role_requests


def get_approvable_resource_requests(person):
    return ServiceResource.objects.filter(state=ServiceResource.APPROVED).active().approvable_by(person)
