from rest_framework import serializers

from plan.api.base import CompactRoleScopeSerializer, ModelSerializer
from plan.roles.models import Role, RoleScope
from plan.services import permissions
from plan.services.models import Service, ServiceMoveRequest
from plan.staff.models import Department, Staff


class ServiceAncestorsSerializer(serializers.Serializer):
    id = serializers.IntegerField()
    slug = serializers.CharField()
    name = serializers.SerializerMethodField()
    name_i18n = serializers.SerializerMethodField()

    def get_name(self, instance):
        if isinstance(instance, dict):
            return instance['name']['ru']

        return instance.name

    def get_name_i18n(self, instance):
        if isinstance(instance, dict):
            return instance['name']

        return {
            'ru': instance.name,
            'en': instance.name_en
        }


class NamedModelSerializer(ModelSerializer):
    name = serializers.SerializerMethodField()
    name_i18n = serializers.SerializerMethodField()

    def get_name(self, instance):
        # todo: Нужно выбирать язык имени на основании хедера Accept-Language из запроса
        return instance.name

    def get_name_i18n(self, instance):
        return {
            'ru': instance.name,
            'en': instance.name_en
        }


class StaffSerializer(serializers.ModelSerializer):

    full_name = serializers.SerializerMethodField()

    def get_full_name(self, staff):
        return staff.get_full_name()

    class Meta:
        model = Staff
        fields = ('login', 'is_dismissed', 'full_name')


class GroupSerializer(serializers.ModelSerializer):

    full_name = serializers.SerializerMethodField()

    def get_full_name(self, group):
        return group.name

    class Meta:
        # ToDo: проверить, что тут только департаменты, а не сервисы или скоупы
        model = Department
        fields = ('id', 'full_name', 'url')


class ServiceSerializer(NamedModelSerializer):
    class Meta:
        model = Service
        fields = ('id', 'slug', 'name', 'name_i18n')


class RoleSerializer(NamedModelSerializer):
    scope = CompactRoleScopeSerializer(queryset=RoleScope.objects.all())

    class Meta:
        model = Role
        fields = ['id', 'name', 'name_i18n', 'service', 'scope', 'code']


class PendingRoleRequestsSerializer(serializers.Serializer):
    idm_request_id = serializers.IntegerField()
    requester = StaffSerializer()
    user = StaffSerializer()
    group = GroupSerializer()
    role = RoleSerializer()
    service_parents = serializers.SerializerMethodField()

    def get_service_parents(self, role_request):
        parents = role_request['service'].get_ancestors(include_self=True)
        serializer = ServiceSerializer(parents, many=True)
        return serializer.data


class PendingMoveRequestsSerializer(serializers.ModelSerializer):
    requester = StaffSerializer()
    approver_outgoing = StaffSerializer()
    approver_incoming = StaffSerializer()

    service = ServiceSerializer()
    destination = ServiceSerializer()

    old_service_parents = serializers.SerializerMethodField(method_name='get_old_parents')
    new_service_parents = serializers.SerializerMethodField(method_name='get_new_parents')
    actions = serializers.SerializerMethodField()

    def get_old_parents(self, move_request):
        old_parents = list(move_request.service.ancestors) + [move_request.service]
        serializer = ServiceAncestorsSerializer(old_parents, many=True)
        return serializer.data

    def get_new_parents(self, move_request):
        new_parents = list(move_request.destination.ancestors) + [move_request.destination, move_request.service, ]
        serializer = ServiceAncestorsSerializer(new_parents, many=True)
        return serializer.data

    def get_actions(self, move_request):
        sender = self.context['sender']
        actions = []

        if not move_request.approver_outgoing and permissions.is_service_responsible(move_request.service, sender):
            actions.extend(['approve_as_outgoing', 'decline_as_outgoing'])

        if not move_request.approver_incoming and permissions.is_service_responsible(move_request.destination, sender):
            actions.extend(['approve_as_incoming', 'decline_as_incoming'])

        return actions

    class Meta:
        model = ServiceMoveRequest
        fields = (
            'actions',
            'approver_incoming',
            'approver_outgoing',
            'destination',
            'id',
            'new_service_parents',
            'old_service_parents',
            'requester',
            'service',
        )
