from django import forms

from rest_framework.views import APIView
from rest_framework.response import Response

from plan.cabinet import helpers
from plan.cabinet.serializers import PendingRoleRequestsSerializer, PendingMoveRequestsSerializer
from plan.services.models import ServiceMoveRequest, ServiceCreateRequest


class OnlyMineFilterForm(forms.Form):
    only_mine = forms.BooleanField(initial=False, required=False)


class RequestCountsView(APIView):
    def get(self, request):
        form = OnlyMineFilterForm(request.query_params)
        if not form.is_valid():
            return Response({'errors': form.errors}, status=400)

        only_mine = form.cleaned_data['only_mine']
        person = request.person
        services = person.get_own_responsible_services(with_descendants=not only_mine)

        role_requests = helpers.get_approvable_role_requests(person, services)
        move_requests = ServiceMoveRequest.objects.approvable(services)
        create_requests = ServiceCreateRequest.objects.approvable(services)
        resource_requests = helpers.get_approvable_resource_requests(person)
        appeals = helpers.get_approvable_appeals(person)

        return Response({
            'request_counts': {
                'role': len(role_requests),
                'service_move': move_requests.count(),
                'service_create': create_requests.count(),
                'resources': resource_requests.count(),
                'appeals': appeals.count(),
            }
        })


class RoleRequestsView(APIView):
    def get(self, request):

        form = OnlyMineFilterForm(request.query_params)
        if not form.is_valid():
            return Response({'errors': form.errors}, status=400)

        only_mine = form.cleaned_data['only_mine']
        services = request.person.get_own_responsible_services(with_descendants=not only_mine)
        role_requests = helpers.get_approvable_role_requests(request.person, services)
        serializer = PendingRoleRequestsSerializer(role_requests, many=True)

        response = {
            'meta': {
                'count': len(role_requests)
            },
            'objects': serializer.data,
        }

        return Response(response)


class ServiceMoveRequestsView(APIView):
    def get(self, request):

        form = OnlyMineFilterForm(request.query_params)
        if not form.is_valid():
            return Response({'errors': form.errors}, status=400)

        only_mine = form.cleaned_data['only_mine']
        services = request.person.get_own_responsible_services(with_descendants=not only_mine)
        move_requests = ServiceMoveRequest.objects.select_related(
            'service', 'destination', 'requester',
            'approver_incoming', 'approver_outgoing',
        ).approvable(services)
        serializer = PendingMoveRequestsSerializer(
            move_requests,
            many=True,
            context={'sender': request.person}
        )

        response = {
            'meta': {
                'count': move_requests.count()
            },
            'objects': serializer.data
        }

        return Response(response)
