from django.contrib import admin
from django.contrib.auth import get_permission_codename
from django.db.models.fields.related import RelatedField


class DefaultAdminMixin:
    """
    class mixin to setup default `raw_id_fields`.
    """
    raw_id_fields = ()

    def __init__(self, model, admin_site, *args, **kwargs):
        self.raw_id_fields = self.setup_raw_id_fields(model)
        super().__init__(model, admin_site, *args, **kwargs)

    def setup_raw_id_fields(self, model):
        return tuple(
            f.name
            for f in model._meta.get_fields()
            if isinstance(f, RelatedField)
        )


class ViewModelAdmin(DefaultAdminMixin, admin.ModelAdmin):

    def get_model_perms(self, request):
        """
        Returns a dict of all perms for this model. This dict has the keys
        ``add``, ``change``, and ``delete`` mapping to the True/False for each
        of those actions.
        """
        return {
            'add': self.has_add_permission(request),
            'change': self.has_change_permission(request),
            'delete': self.has_delete_permission(request),
            'view': self.has_view_permission(request),
        }

    def has_view_permission(self, request, obj=None):
        opts = self.opts
        codename = get_permission_codename('view', opts)
        return request.user.has_perm("%s.%s" % (opts.app_label, codename))

    def has_change_permission(self, request, obj=None):
        return super(ViewModelAdmin, self).has_change_permission(request, obj) or self.has_view_permission(request)

    def get_readonly_fields(self, request, obj=None):
        if super(ViewModelAdmin, self).has_change_permission(request, obj):
            return []

        return [field.name for field in self.model._meta.fields + self.model._meta.many_to_many]
