import yenv
from django.db import models
from django.db.models.fields.related_descriptors import ForwardManyToOneDescriptor


class RelationNotLoaded(Exception):
    pass


class StrictForwardManyToOne(ForwardManyToOneDescriptor):
    def __get__(self, instance, cls=None):
        if yenv.type != 'development.unittest':
            return super().__get__(instance, cls)

        if instance is None:
            return self

        if getattr(instance, self.field.attname) is None:
            return None

        try:
            return getattr(instance, self.cache_name)
        except AttributeError:
            raise RelationNotLoaded(
                'Relation `{rel}` not loaded. Use `select_related` or '
                '`fetch_{rel}`'.format(rel=self.field.name)
            )

    def explicit_get(self, instance, cls=None):
        return super().__get__(instance, cls)


class StrictForeignKey(models.ForeignKey):
    def contribute_to_class(self, cls, name, **kwargs):
        super().contribute_to_class(cls, name, **kwargs)
        #  Override the descriptor defined by ForeignObject
        descriptor = StrictForwardManyToOne(self)
        setattr(cls, self.name, descriptor)
        #  Add a method so you don't always have to use select_related
        fetch_name = 'fetch_{rel}'.format(rel=self.name)
        setattr(cls, fetch_name, lambda inst: descriptor.explicit_get(inst))
