import logging
import waffle

from django.conf import settings
from django.contrib.auth.models import AnonymousUser
from django.db.utils import OperationalError as DjangoOperationalError
from django.http import HttpResponse
from django.utils import translation
from django_replicated.middleware import ReadOnlyMiddleware, ReplicationMiddleware
from django_replicated.utils import routers
from django_yauth.authentication_mechanisms.tvm.request import (
    TvmImpersonatedRequest,
    TvmServiceRequest,
)
from psycopg2 import OperationalError as PGOperationalError

from plan.common.person import Person
from plan.common.internal_roles import get_internal_roles
from plan.common.views import handler404
from plan.common.utils.auth import TVMUser
from plan.common.utils.replicated import is_in_manual_read_only
from plan.common.utils.i18n import get_default_language
from plan.common.utils.tasks import retry_func
from plan.staff.models import Staff

INTRANET_HOST = '.yandex-team.ru'

log = logging.getLogger(__name__)


class AbcReadOnlyMiddleware(ReadOnlyMiddleware):
    """
    При отвалившемся мастере выставляем флаг read_only и переводим
    все запросы на слейв
    """

    @retry_func(exceptions=(DjangoOperationalError, PGOperationalError))
    def is_service_read_only(self):
        is_read_only = super(AbcReadOnlyMiddleware, self).is_service_read_only()
        is_manual_read_only = False
        if is_read_only:
            log.info('Service is in automatic read-only')
        else:
            is_manual_read_only = is_in_manual_read_only()
            if is_manual_read_only:
                log.info('Service is in manual read-only')
        return is_read_only or is_manual_read_only

    def process_response(self, request, response):
        response['X-ABC-READONLY'] = str(getattr(request, 'service_is_readonly', False)).lower()
        response['X-ABC-VERSION'] = settings.VERSION
        return response


class AbcReplicationMiddleware(ReplicationMiddleware):
    """Отличается от базовой только возможностью вручную включить режим read_only,
        для чего используется свойство forced_state"""

    def process_request(self, request):
        self.forced_state = None
        readonly_flag = is_in_manual_read_only()
        if readonly_flag:
            self.forced_state = 'slave'
        result = super(AbcReplicationMiddleware, self).process_request(request)
        request.db_forced_state = self.forced_state
        request.db_state = routers.state()
        request._replication_middleware_processed = True
        return result


class CutEmptyXForwardedHost(object):
    """
    Нужно только для стендов (unstable).
    Чтобы на бэкенде стендов генерился правильные абсолютные ссылки
    (на стенд фронтенда), нужно передавать в django X-Forwarded-Host.
    Однако хочется, чтобы бэкенд не фейлился, если придти напрямую, а не через
    фронт, а у него получается пустой
    X-Forwarded-Host -> пустой host -> SuspiciousOperation

    Я не смог заставить заработать в nginx конструкцию
        set $tmp_x_forwarded_host $host;
        if ($http_x_forwarded_host) {
            set $tmp_x_forwarded_host $http_x_forwarded_host;
        }
        uwsgi_param HTTP_X_FORWARDED_HOST $tmp_x_forwarded_host;
    (хотя отчасти она работала, но что-то все-таки не вышло)

    Вместо этого я здесь удаляю HTTP_X_FORWARDED_HOST если он пришел пустой.
    """
    def process_request(self, request):
        if not settings.USE_X_FORWARDED_HOST:
            return

        if 'HTTP_X_FORWARDED_HOST' not in request.META:
            return

        if not request.META['HTTP_X_FORWARDED_HOST']:
            del request.META['HTTP_X_FORWARDED_HOST']


class I18NMiddleware(object):
    """
        Установка языка вручную
    """

    def process_request(self, request):
        if 'HTTP_ACCEPT_LANGUAGE' in request.META:
            user_lang = request.META['HTTP_ACCEPT_LANGUAGE'].split(',')[0]

            if user_lang in [x[0] for x in settings.LANGUAGES]:
                lang = user_lang
            else:
                lang = get_default_language()

            translation.activate(lang)
            request.LANGUAGE_CODE = lang

    def process_response(self, request, response):
        language = translation.get_language()
        if 'Content-Language' not in response:
            response['Content-Language'] = language
        return response


class CORSMiddleware(object):
    def process_response(self, request, response):
        if (request.META.get('HTTP_ORIGIN')
                and request.META['HTTP_ORIGIN'].endswith(INTRANET_HOST)
                and request.META['HTTP_ORIGIN'].startswith('https://')):
            response['Access-Control-Allow-Origin'] = request.META['HTTP_ORIGIN']
            response['Access-Control-Allow-Credentials'] = 'true'
        return response


class UserMiddleware(object):

    def process_request(self, request):
        yauser = request.yauser
        staff = None
        tvm_service_ticket = None
        if yauser.is_authenticated():

            has_role = self.check_tvm_role(yauser, settings.ABC_API_ACCESS_ROLE)
            if not has_role:
                log.warning(f'Request from {yauser.service_ticket.src} without permission {settings.ABC_API_ACCESS_ROLE} in abc_api was made')
                if waffle.switch_is_active('restrict_tirole'):
                    return HttpResponse(f'You must have role {settings.ABC_API_ACCESS_ROLE} in abc_api system to access ABC api', status=403)

            if isinstance(yauser, TvmImpersonatedRequest):
                staff = Staff.objects.select_related('user', 'department').get(uid=yauser.uid)
                tvm_service_ticket = yauser.service_ticket
            elif isinstance(yauser, TvmServiceRequest):
                tvm_service_ticket = yauser.service_ticket
                user_uid = request.META.get('HTTP_X_UID')
                if user_uid and int(tvm_service_ticket.src) in settings.SERVICE_TVM_ID_ALLOWED_FOR_USERS:
                    staff = Staff.objects.select_related('user', 'department').get(uid=user_uid)
            else:
                staff = Staff.objects.select_related('user', 'department').get(login=yauser.get_username())
        request.tvm_service_ticket = tvm_service_ticket
        request.tvm_user_ticket = request.META.get('HTTP_X_YA_USER_TICKET')
        if staff:
            request.user = staff.user
            request.person = Person(staff)
        else:
            request.user = AnonymousUser()
            if request.tvm_service_ticket:
                request.user = TVMUser()

    def check_tvm_role(self, yauser, role):

        if getattr(yauser.authenticated_by, 'mechanism_name', None) != 'tvm':
            return True

        if waffle.switch_is_active('check_tirole'):
            tvm_client = yauser.authenticated_by.tvm
            return tvm_client.check_service_role(
                yauser.service_ticket,
                f'/role/{role}/',
            )
        else:
            return True


class TvmAccessMiddleware(object):

    IDM_PREFIXES = [
        '/' + settings.IDM_URL_PREFIX,
        '/' + settings.IDM_EXT_URL_PREFIX,
    ]

    def process_regular_view(self, request, view_func):
        view_class = getattr(view_func, 'cls', None)
        if hasattr(request, 'person'):
            # пришли с сервис+юзер тикетом
            return
        elif view_class and hasattr(view_class, 'TVM_ALLOWED_METHODS'):
            if request.method not in view_class.TVM_ALLOWED_METHODS:
                return HttpResponse('You must be authenticated as user', status=403)
        else:
            return HttpResponse('You must be authenticated as user', status=403)

    def process_idm_view(self, request):
        if request.tvm_service_ticket.src != settings.IDM_TVM_ID:
            return HttpResponse('You are not IDM', status=403)

    def process_view(self, request, view_func, view_args, view_kwargs):
        if not request.tvm_service_ticket:
            return
        if any(request.path.startswith(prefix) for prefix in self.IDM_PREFIXES):
            return self.process_idm_view(request)
        return self.process_regular_view(request, view_func)


class CanEditPermissionMiddleware(object):

    def process_view(self, request, view_func, view_args, view_kwargs):
        is_anonymous = isinstance(request.user, AnonymousUser)
        has_tvm = request.tvm_service_ticket is not None
        if (is_anonymous and has_tvm) or request.user.is_superuser:
            return
        if request.method in ['POST', 'PUT', 'PATCH', 'DELETE']:
            if not request.user.has_perm('internal_roles.can_edit'):
                return HttpResponse('Your permissions are not enough', status=403)


class AcceptWithPermissionsOnlyMiddleWare(object):
    def process_view(self, request, view_func, view_args, view_kwargs):
        is_anonymous = isinstance(request.user, AnonymousUser)
        has_tvm = request.tvm_service_ticket is not None
        whitelisted = any(request.path.startswith(prefix) for prefix in settings.URL_PREFIX_WHITELIST)

        if (is_anonymous and has_tvm) or request.user.is_superuser or whitelisted:
            return

        view_class = getattr(view_func, 'cls', None) or getattr(view_func, 'view_class', None)
        proceeding_permission = getattr(view_class, '_permissions_to_proceed', 'can_edit')
        user_perms = get_internal_roles(request.user)

        if not (proceeding_permission in user_perms):
            return HttpResponse('Your permissions are not enough', status=403)


class DisablingUrlMiddleWare:
    def process_view(self, request, view_func, view_args, view_kwargs):
        """
        Для подготовки к выпиливанию ручек.
        Ручку достаточно указать в settings как METHOD:namespaces:url_name
        Возвращает ошибку 404, если ручка указана как устаревшая.

        В process_request еще нет атрибута resolver_match, поэтому используем process_view
        """

        method = request.method
        view_name = request.resolver_match.view_name
        if f'{method}:{view_name}' not in settings.OBSOLETE_URLS:
            return

        return handler404(request)
