from django.db import models
from django.utils import timezone

from closuretree.models import ClosureModel


class TimestampedModel(models.Model):

    created_at = models.DateTimeField(auto_now_add=True)
    updated_at = models.DateTimeField(auto_now=True)

    class Meta:
        abstract = True


def update_model_instance(saved_obj, real_obj, fields=None):
    if fields is None:
        fields = {field.name for field in saved_obj._meta.fields} - {'id'}

    update_fields = []
    for field in fields:
        saved = getattr(saved_obj, field)
        real = getattr(real_obj, field)
        if saved != real:
            setattr(saved_obj, field, real)
            update_fields.append(field)

    if update_fields:
        saved_obj.save(update_fields=update_fields)


def update_model_fields(obj, fields):
    changed_fields = []
    for key, new_value in fields.items():
        old_value = getattr(obj, key)
        if old_value != new_value:
            setattr(obj, key, new_value)
            changed_fields.append(key)
            if key in {'parent', 'parent_id'} and isinstance(obj, ClosureModel):
                changed_fields.append('level')
    if changed_fields:
        obj.save(update_fields=changed_fields)


def date_from_str(str_date):
    """Конвертирует str_date в datetime.date, если str_date != None, иначе возвращает None"""
    if str_date is None:
        return str_date
    return timezone.datetime.strptime(str_date, '%Y-%m-%d').date()


def str_from_date(date):
    """Конвертирует date в datetime.date, если date != None, иначе возвращает None"""
    if date is None:
        return date
    return timezone.datetime.strftime(date, '%Y-%m-%d')
