from django.db.models import Q
from django.utils.functional import cached_property

from plan.duty.models import Schedule, Order
from plan.roles.models import Role, merge_allowed_roles
from plan.services.models import Service, ServiceMember
from plan.staff.models import Staff


class Person(object):

    def __init__(self, staff):
        assert isinstance(staff, Staff), 'Should be Staff instance, not %s' % type(staff)
        self.staff = staff
        self._is_responsible_cache = {}

    def __getattr__(self, item):
        return getattr(self.staff, item)

    def __repr__(self):
        return 'Person proxy for %r' % self.staff

    def __eq__(self, other):
        return self.staff == other.staff

    @cached_property
    def own_services(self):
        """ Сервисы, в которых состоит человек
        """
        return set(role.service for role in self.own_roles)

    @cached_property
    def own_roles(self):
        """ Роли человека в сервисах
        """
        return ServiceMember.objects.filter(staff=self.staff).select_related('service', 'role')

    def get_own_responsible_services(self, with_descendants=False):
        """ Сервисы, в которых человек управляющий
        """
        return Service.objects.with_responsible(self, with_descendants=with_descendants)

    @cached_property
    def head_roles(self):
        """ Инстансы ролей руководителей
        """
        return Role.objects.filter(code__in=(Role.EXCLUSIVE_OWNER, Role.DEPUTY_OWNER))

    def is_member_of_team(self, service, with_ancestors=True):
        """ Является ли человек участником команды
        """
        for own_service in self.own_services:
            if with_ancestors:
                if service.is_ancestor_of(own_service, include_self=True):
                    return True
            else:
                if service == own_service:
                    return True

        return False

    def has_role(self, service, role, with_descendants=True):
        """ Есть ли у человека роль
        """
        for membership in self.own_roles:
            if membership.role == role:
                if with_descendants:
                    if membership.service.is_ancestor_of(service, include_self=True):
                        return True
                else:
                    if membership.service == service:
                        return True

        return False

    def is_responsible(self, service):
        """ Является ли человек ответственным за сервис
        """
        if service.id not in self._is_responsible_cache:
            value = False
            responsible_services = self.get_own_responsible_services()
            if responsible_services and service.is_descendant_of_others(responsible_services, include_self=True):
                value = True

            self._is_responsible_cache[service.id] = value

        return self._is_responsible_cache[service.id]

    def can_modify_duty_in_service(self, service):
        """
        Может ли человек управлять дежурствами в сервисе
        """
        user_is_duty_manager = service.members.filter(
            staff=self.staff,
            role__code=Role.RESPONSIBLE_FOR_DUTY,
        ).exists()
        return user_is_duty_manager or self.is_responsible(service) or self.staff.user.is_superuser

    def is_schedule_member(self, schedule):
        """Является ли человек участником графика
        """
        if schedule.algorithm == Schedule.MANUAL_ORDER:
            return Order.objects.filter(schedule=schedule, staff=self.staff).exists()
        else:
            return (
                ServiceMember.objects
                .filter(Q(service=schedule.service, staff=self.staff) & schedule.get_role_q())
                .exists()
            )

    def is_head(self, service):
        """ Является ли человек руководителем или заместителем
        """
        return any(self.has_role(service, role) for role in self.head_roles)

    def is_consumer_agent(self, service, resource_type):
        if self.user.is_superuser:
            return True

        for permission in resource_type.consumer_permissions.all():
            perm_codename, perm_app, _ = permission.natural_key()
            perm_key = '{}.{}'.format(perm_app, perm_codename)
            if self.user.has_perm(perm_key):
                return True

        allowed_roles = merge_allowed_roles(resource_type.consumer_roles, resource_type.consumer_scopes, service)
        return any(self.has_role(service, role, with_descendants=False) for role in allowed_roles)

    def is_supplier_agent(self, resource_type):
        if self.user.is_superuser:
            return True

        service = resource_type.supplier
        allowed_roles = merge_allowed_roles(resource_type.supplier_roles, resource_type.supplier_scopes, service)
        return any(self.has_role(service, role, with_descendants=False) for role in allowed_roles)
