import urllib.parse

from django.http import QueryDict
from django.conf import settings
from django.contrib.auth import REDIRECT_FIELD_NAME
from django.contrib.auth.models import AnonymousUser

from plan.exceptions import HttpRedirect


def check_authentication(request):
    """
    Функция, которая заменяет декоратор login_required
    Отличия от login_required:
      * Это не декоратор
      * Максимально упрощена логика
      * Не возвращаем HttpResponseRedirect, а рейзим HttpRedirect
    """
    if not request.user.is_authenticated():
        path = request.build_absolute_uri()
        login_url = settings.LOGIN_URL
        scheme, netloc, path, params, query, fragment = (
            list(urllib.parse.urlparse(login_url))
        )
        querystring = QueryDict(query, mutable=True)
        querystring[REDIRECT_FIELD_NAME] = path
        updated_query = querystring.urlencode(safe='/')

        updated_login_url = urllib.parse.urlunparse(
            [scheme, netloc, path, params, updated_query, fragment]
        )
        raise HttpRedirect(updated_login_url)


def get_full_access_permissions_queryset():
    from django.contrib.auth.models import Permission
    permissions = settings.ABC_INTERNAL_ROLES_PERMISSIONS.get(settings.FULL_ACCESS)
    return Permission.objects.filter(
        codename__in=permissions,
        content_type__app_label='internal_roles'
    )


class TVMUser(AnonymousUser):
    _user_permissions = get_full_access_permissions_queryset()

    def get_all_permissions(self):
        permissions = self._user_permissions.values_list('content_type__app_label', 'codename')
        return {f'{app_label}.{codename}' for app_label, codename in permissions}

    def has_perm(self, perm):
        if perm in self.get_all_permissions():
            return True

        return False
