from itertools import product
import logging

from django.db import transaction


logger = logging.getLogger(__name__)


def _create_closure_tree_nodes(model_class, ids):
    logger.info('Creating closure nodes for objects {}..{}'.format(ids[0], ids[-1]))
    with transaction.atomic():
        nodes = (model_class._closure_model(parent_id=pk, child_id=pk, depth=0) for pk in ids)
        model_class._closure_model.objects.bulk_create(nodes)


def regenerate_closure_tree(model_class, batch_size=5000):
    logger.info('%s objects to create', model_class.objects.count())
    ids = model_class.objects.values_list('pk', flat=True).iterator()

    logger.info('Cleaning existing closure nodes...')
    model_class._closure_model.objects.all().delete()

    counter = 0
    creation_queue = []

    # создаем ноды в closuretree модели батчами по 1000
    for _id in ids:
        creation_queue.append(_id)
        counter += 1
        if counter >= batch_size:
            _create_closure_tree_nodes(model_class, creation_queue)
            counter = 0
            del creation_queue
            creation_queue = []

    # подчищаем оставшиеся от последней итерации,
    # (общее количество объектов вряд ли кратно {{BATCH_SIZE}})
    if creation_queue:
        _create_closure_tree_nodes(model_class, creation_queue)

    for instance in model_class.objects.iterator():
        create_closure_links(instance)


def create_closure_links(model_instance):
    logger.info(
        'Building closure links for %s.%s %s',
        model_instance._meta.app_label,
        model_instance._meta.model_name,
        model_instance.pk
    )

    parents = model_instance._closure_model.objects.filter(
        child__pk=model_instance._closure_parent_pk).values("parent", "depth")
    children = model_instance._closure_model.objects.filter(
        parent__pk=model_instance.pk).values("child", "depth")

    links = [
        model_instance._closure_model(
            parent_id=parent['parent'],
            child_id=child['child'],
            depth=parent['depth']+child['depth']+1
        )
        for parent, child in product(parents, children)
    ]

    if not links:
        logger.info(
            '%s.%s %s had no relationships to generate',
            model_instance._meta.app_label,
            model_instance._meta.model_name,
            model_instance.pk
        )
        return

    with transaction.atomic():
        model_instance._closure_model.objects.bulk_create(links)
