from collections.abc import Mapping
from functools import reduce
from itertools import chain, groupby, repeat
from operator import attrgetter

from django.http import QueryDict


_NOT_SET = object()


def get_multiple(mapping, keys, default=_NOT_SET):
    """
    Вернуть список значений по списку ключей для словаря.
    Если ключа нет в словаре по умолчанию в результатах он не появится. Можно
    передать `default` и тогда это значение будет использовано для ненайденного
    ключа.
    Порядок соответствует порядку ключей в `keys`.
    """
    values = []
    for key in keys:
        if default is _NOT_SET and key not in mapping:
            continue
        values.append(mapping.get(key, default))
    return values


def pop_multiple(mapping, keys, default=_NOT_SET):
    """
    Попнуть список значений по списку ключей для словаря.
    Если ключа нет в словаре по умолчанию в результатах он не появится. Можно
    передать `default` и тогда это значение будет использовано для ненайденного
    ключа.
    Порядок соответствует порядку ключей в `keys`.
    """
    values = []
    for key in keys:
        if default is _NOT_SET and key not in mapping:
            continue
        values.append(mapping.pop(key, default))
    return values


def filter_by_conditions(mapping):
    """
    `mapping` -- dict с без-аргументными функциями в качестве значений
    Возвращает ключи, для которых соответствующее значение-функция вернет True.
    TODO: действительно ли это реиспользуемо и нельзя сделать проще?
    TODO: тесты
    """
    # TOTHINK:
    # pairs: return filter(lambda (key, value): value(), mapping.iteritems())
    # dict: return {key: value for key, value in mapping.iteritems() if value()}
    return [key for key in mapping if mapping[key]()]


def filter_dict(mapping, keys):
    """
    Возвращает dict, содержащий только ключи из списка `keys`
    и значения из исходного.
    Если в `keys` есть ключи, которых нет в `mapping`, то эти ключи не
    попадают в результирующий словарь.
    """
    keys = [key for key in keys if key in mapping]
    return {
        key: value
        for key, value in zip(keys, get_multiple(mapping, keys))
    }


def updated_dict(mapping, **kwargs):
    """
    Возвращает новый dict, сделанный из mapping с добавленными
    ключами и значениями из kwargs
    """
    new_dict = mapping.copy()
    new_dict.update(kwargs)
    return new_dict


def groupby_as_dict(iterable, keys):
    """
    Сгруппировать последовательность сущностей `iterable` с ключами из key,
    отсортированную по ключам из `keys`.
    Вернуть словарь с ключами, составленными из значений ключей и списками
    сгруппированных сущностей в качестве значений.
    """
    return dict(
        (key, list(gen)) for (key, gen)
        in groupby(iterable, attrgetter(*keys))
    )


def convert_mapping_to_iterable(mapping, key_name, value_name):
    """
    Преобразовать словарь в список словарей, где ключи `mapping` кладутся в
    список в ключи с именем `key_name`, а значения в ключи с именем.
    Порядок диктов в последовательности не определен.
    `value_name`.
    :return: list
    """
    return [
        {key_name: key, value_name: value}
        for key, value in mapping.items()
    ]


def revert_mapping(mapping):
    """
    Развернуть словарь.
    """
    return {value: key for key, value in mapping.items()}


def revert_mapping_of_iterables(mapping):
    """
    Convert dict of iterables like
    {
        1: ['x', 'y'],
        2: ['z', 'u']
    }
    to dict
    {
        'x': 1,
        'y': 1,
        'z': 2,
        'u': 2,
    }
    Items in `mapping` values lists must be unique across dict.
    """
    reverted_pairs = (
        list(zip(ids, repeat(type_)))
        for type_, ids in mapping.items()
    )
    flat_pairs = chain.from_iterable(reverted_pairs)
    return dict(flat_pairs)


def revert_mapping_of_iterables_list_values(mapping):
    """
    Naming is hard :(

    Convert dict of iterables like
    {
        1: ['x', 'y'],
        2: ['z', 'y']
    }
    to dict
    {
        'x': [1],
        'y': [1, 2],
        'z': [2],
    }
    Like revert_mapping_of_iterables, but result value are lists and
    initial `mapping` values in lists can not be unique across dict.
    """
    reverted_pairs = (
        list(zip(ids, repeat(type_)))
        for type_, ids in mapping.items()
    )
    flat_pairs = chain.from_iterable(reverted_pairs)

    result_map = {}
    for key, value in flat_pairs:
        list_value = result_map.setdefault(key, [])
        list_value.append(value)

    return result_map


def has_key_by_sequence(mapping, key_sequence):
    """
    Есть ли во вложенном словаре все ключи по пути key_sequence.
    >>> has_key_by_sequence({'a': {'b': 42}}, ['a', 'b'])
    ... True
    >>> has_key_by_sequence({'a': {'b': 42}}, ['a', 'b', 'c'])
    ... False
    # TODO: наверное как-то элегантнее можно придумать.
    """
    def go_deeper(mapping, keys):
        for key in keys:

            if not isinstance(mapping, Mapping):
                # проверка ключа, а это даже не словарь!
                yield False
                return

            yield key in mapping
            # key error не будет, потому что all ленив
            # и сдаст нас при первом False
            mapping = mapping[key]

    return all(go_deeper(mapping, key_sequence))


def get_by_sequence(mapping, key_sequence, default=None):
    """
    Получить значение из словаря, состоящего из вложенных словарей.
    >>> get_by_sequence({'a': {'b': 42}}, ['a', 'b'])
    ... 42
    >>> get_by_sequence({'b': 42}, ['b'])
    ... 42
    """
    if not has_key_by_sequence(mapping, key_sequence):
        return default

    return reduce(
        lambda mapping, key: mapping[key],
        key_sequence,
        mapping,
    )


def set_by_sequence(mapping, key_sequence, value):
    """
    Записать значение в словарь, состоящий из вложенных словарей.
    >>> d = {'a': {'b': 42}}
    >>> set_by_sequence(d, ['a', 'b'], 300)
    >>> print d.items()
    >>> [('a', {'b': 300})]
    # TODO: сделать, чтобы создавались промежуточные ключи, если нужно.
    """
    head, last_element = key_sequence[:-1], key_sequence[-1]
    value_owner = get_by_sequence(mapping, head)
    value_owner[last_element] = value


def dict_to_querydict(dict_):
    """
    Конвертирует обычный dict в QueryDict
    """
    query_dict = QueryDict('', mutable=True)
    for key, value in dict_.items():
        if isinstance(value, list):
            query_dict.setlist(key, value)
        elif callable(value):
            query_dict[key] = value()
        else:
            query_dict[key] = value
    return query_dict


class DotAccessedDict(dict):
    """
    Словарь, к полям которого можно доступаться через точку.
    Вложенные словари также обретают это свойство.
    """
    def __init__(self, *args, **kwargs):
        super(DotAccessedDict, self).__init__(*args, **kwargs)

        for key, value in self.items():
            if isinstance(value, Mapping):
                self[key] = DotAccessedDict(value)

    def __getattr__(self, item):
        try:
            return self.__getitem__(item)
        except KeyError:
            raise AttributeError(item)

    def __setattr__(self, key, value):
        """
        BTW inconsistency http://bugs.python.org/issue14658
        __setattr__ = dict.__setitem__ doesn't work
        """
        self.__setitem__(key, value)

    def __setitem__(self, key, value):
        if isinstance(value, Mapping):
            value = DotAccessedDict(value)
        super(DotAccessedDict, self).__setitem__(key, value)
