import datetime
from functools import partial

from django.conf import settings
import pytz


def datetime_isoformat(dt):
    return dt.strftime('%Y-%m-%dT%H:%M:%SZ')


def datetime_isoformat_with_microseconds(dt):
    return dt.strftime('%Y-%m-%dT%H:%M:%S.%fZ')


def datetime_tzformat_with_microseconds(dt):
    timezone = pytz.timezone(settings.TIME_ZONE)
    dt = dt.astimezone(timezone)
    return dt.isoformat()


# TODO: нужно одинаково назвать
def isoformat_to_date(str):
    return datetime.datetime.strptime(str, '%Y-%m-%d').date()


def datetime_from_str(dt_str, format_str='%Y-%m-%dT%H:%M:%S'):
    return datetime.datetime.strptime(dt_str, format_str)


def datetime_to_str(dt, format_str='%Y-%m-%dT%H:%M:%S'):
    return dt.strftime(format_str)


# TODO: Use plan.common.utils.periods.Quarter.from_date instead.
def date_to_quarter(dt):
    """Возвращает тапл из квартала и года по заданной дате"""
    return 1 + (dt.month - 1) / 3, dt.year


def quarter_to_first_month(q, year):
    """Переводит Квартал и Год в дату (берем первый месяц квартала)"""
    try:
        # первое число квартала
        return datetime.date(int(year), int(q) * 3 - 2, 1)
    except ValueError:
        return None


def quarter_to_last_month(q, year):
    """Переводит Квартал и Год в дату (берем последний месяц квартала)"""
    try:
        return datetime.date(int(year), int(q) * 3, 1)
    except ValueError:
        return None


def get_quarter_boundaries(dt):
    """
    Возвращает кортеж из границ квартала, в котором находится `dt`.

    >>> get_quarter_boundaries(datetime.date(2011, 2, 18))
    (datetime.date(2011, 1, 1), datetime.date(2011, 4, 1))
    >>> get_quarter_boundaries(datetime.date(2011, 4, 1))
    (datetime.date(2011, 4, 1), datetime.date(2011, 7, 1))
    >>> get_quarter_boundaries(datetime.date(2011, 9, 30))
    (datetime.date(2011, 7, 1), datetime.date(2011, 10, 1))
    """
    right = next_quarter(dt)
    left = get_relative_month(right, -3)
    return left, right


# TODO: реализовано в plan.common.utils.periods для месяца и квартала
def get_relative_month(dateobj, offset=1):
    """
    >>> get_relative_month(datetime.date(2011, 2, 18), -1)
    datetime.date(2011, 1, 1)
    >>> get_relative_month(datetime.date(2011, 2, 18), 0)
    datetime.date(2011, 2, 1)
    >>> get_relative_month(datetime.date(2011, 2, 18), 1)
    datetime.date(2011, 3, 1)
    >>> get_relative_month(datetime.date(2011, 1, 1), 25)
    datetime.date(2013, 2, 1)
    >>> get_relative_month(datetime.date(2011, 1, 1), -25)
    datetime.date(2008, 12, 1)
    """
    current_year, current_month = dateobj.year, dateobj.month
    month = current_month + offset
    mdiv, mquot = divmod(month - 1, 12)
    year = current_year + mdiv
    month = mquot + 1
    return datetime.date(year=year, month=month, day=1)


next_month = get_relative_month
prev_month = partial(get_relative_month, offset=-1)


def get_relative_quarter(dt, offset=1):
    quarter_end = quarter_to_last_month(*date_to_quarter(dt))
    return get_relative_month(quarter_end, offset * 3)


next_quarter = get_relative_quarter
prev_quarter = partial(get_relative_quarter, offset=-1)


def in_period(dt, period):
    start_date, end_date = period

    start_date = start_date or dt
    end_date = end_date or dt

    return start_date <= dt <= end_date


def date_range(start_date, end_date, delta=None, inclusive=False):
    cur = start_date
    delta = delta or datetime.timedelta(1)

    if inclusive:
        def test(a, b):
            return a <= b
    else:
        def test(a, b):
            return a < b

    while test(cur, end_date):
        yield cur
        cur += delta


def end_date_cmp(project, date_for_month):
    """
    Определяет, заканчивается ли проект:
      * в течении месяца date_for_month (0);
      * до начала месяца date_for_month (-1);
      * после месяца date_for_month (1).
    """
    month = datetime.date(year=date_for_month.year, month=date_for_month.month, day=1)

    if project.end_date > next_month(month):
        return 1
    elif project.end_date <= month:
        return -1
    else:
        return 0


# TODO: реализовано в plan.common.utils.periods для месяца и квартала
def get_current_month():
    return datetime.date.today().replace(day=1)


def date_to_datetime(date_value):
    """
    Преобразовать дату в датувремя, добавив нули.
    Необходимо для хранения дат в монге, где нет дат без времени.
    """
    return datetime.datetime.combine(
        date_value,
        datetime.time(0, 0, tzinfo=pytz.utc)
    )


def datetime_to_date(datetime_value):
    """
    Преобразовать датувремя в дату, откинув информацию о времени.
    Необходимо для полей в монге, в которых лежит датавремя, но по логике это
    дата.
    """
    return datetime_value.date()
