from collections import OrderedDict, namedtuple


class EnumChoice(object):
    _counter = 0

    def __init__(self, code, **kwargs):
        self.code = code
        self.params = kwargs

        self.i = EnumChoice._counter
        EnumChoice._counter += 1

    def __hash__(self):
        return self.i


class Enum(object):
    def __init__(self, choices, groups):
        self._params = OrderedDict()

        for attr, choice in sorted(choices.items(), key=lambda c: c[1].i):
            setattr(self, attr, choice.code)

            cls = namedtuple('EnumParams', list(choice.params.keys()))
            self._params[choice.code] = cls(**choice.params)

        for attr, values in groups.items():
            setattr(self, attr, values)

    def __getitem__(self, key):
        return self._params.__getitem__(key)

    def __iter__(self):
        return iter(list(self._params.keys()))


class EnumMeta(type):
    def __new__(mcs, name, bases, attributes):
        cls = type(name, bases, attributes)

        choices = {}
        groups = {}

        for attribute in attributes:
            val = getattr(cls, attribute)
            if isinstance(val, EnumChoice):
                choices[attribute] = val
            elif (isinstance(val, (set, frozenset))
                    and all([isinstance(x, EnumChoice) for x in val])):
                groups[attribute] = frozenset([choice.code for choice in val])

        return Enum(choices, groups)
