from django import forms
from django.core.validators import EMPTY_VALUES
from django.utils.encoding import smart_text


class StateOrGroupField(forms.MultipleChoiceField):
    """
    Это поле умеет принимать список статусов или название группы статусов.
    Приоритет имеет название группы.
    Также через это поле можно получить эффективный список выбранных статусов
    для фильтрации, вне зависимости от того, статусы или группа в него пришли.
    """

    def __init__(self, *args, **kwargs):
        self.states = tuple(kwargs.pop('states'))
        self.groups = tuple(kwargs.pop('groups'))
        kwargs['choices'] = self.states + self.groups
        super(StateOrGroupField, self).__init__(*args, **kwargs)

    def get_states(self, cleaned):
        for id, group in self.groups:
            if id in cleaned:
                return group['states']
        return set(cleaned)


class FooSeparatedValuesField(forms.CharField):
    """Поле, представляющее список строковых значений, разделенных разделителем.

    Преобразуется в список строк.
    Пример: //some_url?fields=id|code|name --> ['id', 'code', 'name']

    Параметр `unicode` определяет будут ли строки в результирующем списке
    юникодными или байтовыми.

    Параметр `permitted_values` принимает список допустимых значений для
    валидации.

    Я нагло скопипастил это из multic.
    """
    default_error_messages = {
        'invalid_values': 'Some of given values are not permitted: %s'
    }

    def __init__(self, *args, **kwargs):
        self.str = kwargs.pop('unicode', True)
        self.permitted_values = kwargs.pop('permitted_values', None)
        self.separator = kwargs.pop('separator', ',')
        super(FooSeparatedValuesField, self).__init__(*args, **kwargs)

    def to_python(self, value):
        if value in EMPTY_VALUES:
            return []
        string_function = smart_text if self.str else str
        return [string_function(data) for data in value.split(self.separator)]

    def validate(self, values_list):
        if self.permitted_values is None:
            return values_list

        diff = set(values_list) - set(self.permitted_values)
        if diff:
            raise forms.ValidationError(
                self.error_messages['invalid_values'] % ', '.join(diff))
        return values_list
