from django.http import QueryDict

from plan.common.utils.collection.mapping import dict_to_querydict


class RequiredFormMixin(object):
    required_css_class = 'b-plan-form__field_required'


class DefaultDataMixin(object):
    """
    Миксин для форм.

    Если в data не были переданые параметры из default_data,
    то значения оттуда будут добавлены в data.
    """
    default_data = {}

    def __init__(self, data, *args, **kwargs):
        if isinstance(data, QueryDict):
            new_data = data.copy()
        else:
            new_data = dict_to_querydict(data)

        default_data = dict_to_querydict(self.default_data)

        # Заполняем входные параметры недостающими дефолтными
        for key, value in default_data.lists():
            if key not in new_data:
                new_data.setlist(key, value)

        super(DefaultDataMixin, self).__init__(new_data, *args, **kwargs)

        # Создаем словарь, который содержит ключи только из self.fields,
        # чтобы сравнить его с default_data на признак дефолтности формы
        filtered_data = QueryDict({}, mutable=True)
        for key, value in new_data.lists():
            if key in self.fields:
                filtered_data.setlist(key, value)

        self.is_default = filtered_data == default_data

    @classmethod
    def apply_default_data(cls):
        for key, value in cls.default_data.items():
            if callable(value):
                cls.default_data[key] = value()


class JSArrayConvertingMixin(object):
    """
    Миксин для django-форм.

    Преобразует ключи, которые пришли в данные формы в виде
    {
        'date[from]': '2012-01-01',
        'date[to]': '2012-01-10',
    }
    в
    {
        'date_to': '2012-01-01',
        'date_from': '2012-01-10',
    }
    Для того, чтобы миксин сработал явно указываем атрибут `js_arrays` у формы.
    Например:
    js_arrays = {
        'date': ['from', 'to']
    }
    Информация из `js_arrays` будет также использована для дегидрации формы.
    """
    js_arrays = {}

    def __init__(self, data, *args, **kwargs):
        data = self.convert_array_keys(data)
        super(JSArrayConvertingMixin, self).__init__(data, *args, **kwargs)

    def convert_array_keys(self, data):
        from django.http import QueryDict

        data_copy = QueryDict('', mutable=True)

        if isinstance(data, QueryDict):
            for key, value in data.lists():
                key = self.convert_key_if_needed(key)
                data_copy.setlist(key, value)

        return data_copy

    def convert_key_if_needed(self, key):
        return self.array_query_params_to_form_fields.get(key, key)

    @property
    def array_query_params_to_form_fields(self):
        mapping = {}
        for array, keys in self.js_arrays.items():
            for key in keys:
                expected_param = self.build_query_param(array, key)
                field_name = self.build_form_field(array, key)
                mapping[expected_param] = field_name
        return mapping

    @staticmethod
    def build_query_param(array_name, key):
        return '{}[{}]'.format(array_name, key)

    @staticmethod
    def build_form_field(array_name, key):
        return '{}_{}'.format(array_name, key)
