import copy
import operator
import collections.abc

from functools import reduce


class SummableDict(dict):
    '''Summable Dict is a Dict supporting addition (by .update())'''
    def __add__(self, other):
        result = copy.copy(self)
        result.update(other)
        return result


D = SummableDict


class HybridValidatorBase(object):
    '''Looks like a form, but it's NOT a form!'''
    validators = []

    def __init__(self, *args, **kwargs):
        self._forms = [f(*args, **kwargs) for f in self.validators]

    @property
    def cleaned_data(self):
        return reduce(operator.add, (D(f.cleaned_data) for f in self._forms))

    @property
    def errors(self):
        def feed_errors():
            for f in self._forms:

                # Form.errors является наследником dict
                if isinstance(f.errors, collections.abc.Mapping):
                    yield D(f.errors)

                # FormSet.errors является list[dict]
                elif isinstance(f.errors, collections.abc.Sequence):
                    for d in f.errors:
                        yield D(d)

        return reduce(operator.add, feed_errors())

    def is_valid(self):
        return all(f.is_valid() for f in self._forms)


def hybrid_validator_factory(*validators):
    return type('HybridValidator', (HybridValidatorBase,),
                {'validators': validators})


def form_errors_as_text(errors):
    field_errors = []
    for key, values in errors.items():
        values = '; '.join(values)
        if key == '__all__':
            field_errors.append(values)
        else:
            field_errors.append(f'{key}: {values}')
    return '; '.join(field_errors)
