from datetime import datetime
import functools
import logging
import time

import ylock

from django.conf import settings

logger = logging.getLogger(__name__)

lock_manager = ylock.create_manager(**settings.YLOCK)

locked_context = lock_manager.lock


def lock(lock_key=None, min_lock_time=30):
    def _decorator(f):
        @functools.wraps(f)
        def _wrapper(*args, **kwargs):
            block = kwargs.pop('_lock_block', False)
            if not settings.LOCKS_ENABLED:
                return f(*args, **kwargs)

            name = f'{f.__module__}.{f.__name__}'
            if lock_key:
                name += lock_key(*args, **kwargs) if callable(lock_key) else lock_key
            with locked_context(name, block=block) as acquired:
                if acquired:
                    start = datetime.utcnow()
                    result = f(*args, **kwargs)
                    execution_time = (datetime.utcnow() - start).total_seconds()
                    if 0 < execution_time < min_lock_time:
                        time.sleep(min_lock_time - execution_time)
                    return result
                else:
                    logger.warning('Lock with name %s already acquired', name)
                    return 'locked'
        return _wrapper
    return _decorator
