from functools import wraps
import types


def memoizable_property(getter):
    key = '_cache_prop_' + getter.__name__

    def getter_wrapper(self, *args, **kwargs):
        if not hasattr(self, key):
            result = getter(self, *args, **kwargs)
            if isinstance(result, types.GeneratorType):
                result = list(result)
            setattr(self, key, result)
            return result
        return getattr(self, key)

    return property(getter_wrapper)


def memoize_in_arg(func):
    """
    Декоратор для кеширования результата вызова функции в ее единственном
    аргументе.
    """

    key = '_cached_result_' + func.__name__

    @wraps(func)
    def wrapper(arg):
        if not hasattr(arg, key):
            result = func(arg)
            if isinstance(result, types.GeneratorType):
                result = list(result)
            setattr(arg, key, result)
        return getattr(arg, key)

    return wrapper


def memoize_in_first_arg(func=None, keygetter=None):
    """
    Декоратор для кеширования результата вызова функции в ее первом
    аргументе. Первый аргумент должен быть позиционным, а не ключевым.
    Возможно определение функции keygetter для получения получения из остальных
    аргументов хэшируемого объекта для использования в качестве ключа в кэше.
    """
    if callable(func):
        return _memoize_in_first_arg(func, None)
    else:
        def deco(func):
            return _memoize_in_first_arg(func, keygetter)
        return deco


def _memoize_in_first_arg(func, keygetter):
    if keygetter is None:
        def keygetter(*args, **kwargs):
            return (args, kwargs)

    cache_name = 'cached_result_' + func.__name__

    @wraps(func)
    def wrapper(first_arg, *args, **kwargs):
        key = keygetter(*args, **kwargs)

        if not hasattr(first_arg, cache_name):
            setattr(first_arg, cache_name, {})

        cache = getattr(first_arg, cache_name)
        hit = cache.get(key, '__missing')
        if hit == '__missing':
            val = func(first_arg, *args, **kwargs)
            cache[key] = val
            return val
        else:
            return hit

    return wrapper
