from django import forms
from django.core import validators
from django.core.exceptions import ValidationError
from django.utils.safestring import mark_safe

from plan.services.models import Service
from plan.staff.models import Department, Staff


MULTIC_ENTITIES = {
    'staff': {'model': Staff, 'name': str},
    'platform': {'model': Service, 'name': str},
    'product': {'model': Service, 'name': str},
    'service': {'model': Service, 'name': str},
    'department': {'model': Department, 'name': str},
}

MODEL_NAME = {
    Staff: 'staff',
}


def reverse_multic_entity(cls):
    for type_ in MULTIC_ENTITIES:
        cell = MULTIC_ENTITIES[type_]
        if cls == cell['model']:
            return type_, cell['name']
    return '', lambda x: ''


class MulticWidget(forms.MultiWidget):
    def __init__(self, attrs=None):
        widgets = (forms.TextInput(attrs=attrs),
                   forms.TextInput(attrs=attrs),
                   forms.TextInput(attrs=attrs),)
        super(MulticWidget, self).__init__(widgets, attrs)

    def decompress(self, value):
        if value:
            model_name = MODEL_NAME[type(value)]
            return [model_name,
                    value.id,
                    MULTIC_ENTITIES[model_name]['name'](value)]

        return [None, None, None]

    def render(self, name, value, attrs=None):
        if not isinstance(value, list):
            value = self.decompress(value)
        output = []
        final_attrs = self.build_attrs(attrs)
        id_ = final_attrs.get('id', None)
        names = ('__type', '__value', '')
        for i, widget in enumerate(self.widgets):
            try:
                widget_value = value[i]
            except IndexError:
                widget_value = None
            if id_:
                final_attrs = dict(final_attrs, id=name + names[i])
            output.append(widget.render(name + names[i], widget_value, final_attrs))
        return mark_safe(self.format_output(output))

    def value_from_datadict(self, data, files, name):
        names = ('__type', '__value', '')
        return [widget.value_from_datadict(data, files, name + names[i])
                for i, widget in enumerate(self.widgets)]


class MulticField(forms.MultiValueField):
    widget = MulticWidget

    def __init__(self, *args, **kwargs):
        fields = (
            forms.CharField(),  # type
            forms.CharField(),  # value (т.е. id)
            forms.CharField(),  # текстовое представление
        )
        super(MulticField, self).__init__(fields, *args, **kwargs)

    def compress(self, data_list):
        if data_list:
            multic_types, obj_value, obj_name = data_list
            if multic_types in validators.EMPTY_VALUES:
                raise ValidationError(self.error_messages['invalid'])
            if obj_value in validators.EMPTY_VALUES:
                return None
            for multic_type in multic_types.split('|'):
                data_type = MULTIC_ENTITIES[multic_type]['model']
                try:
                    return data_type.objects.get(pk=obj_value)
                except data_type.DoesNotExist:
                    pass
        return None
