from urllib.parse import urljoin
from typing import Tuple, List
from django.conf import settings

from plan.common.utils.http import Session
from plan.common.utils.startrek import create_comment
from plan.oebs.constants import ERRORS


class NoRelatedIssueFound(Exception):
    pass


class OkClient:
    def __init__(self):
        self.token = settings.OK_TOKEN
        self.host = settings.OK_API_URL

    def create_request(self, oebs_agreement, approvers: List[Tuple[str]]) -> str:
        if oebs_agreement.ok_id:
            return oebs_agreement.ok_id

        if not oebs_agreement.issue:
            raise NoRelatedIssueFound()

        data = {
            'type': 'tracker',
            'object_id': oebs_agreement.issue,
            'text': 'Согласование изменения OEBS дерева',
            'stages': [
                {
                    'need_all': False,
                    'stages': [
                        {'approver': login}
                        for login in approvers_group
                    ]
                }
                for approvers_group in approvers
            ],
            'groups': ['svc_abc', 'yandex_fin_control_acc_oebs'],
            'is_parallel': True,
        }
        url = urljoin(self.host, 'api/approvements/')

        with Session(oauth_token=self.token) as session:
            response = session.post(url=url, json=data)
            if not response.ok:
                oebs_agreement.fail(
                    error=ERRORS.ABC_ERROR,
                    message={
                        'error': str(response.content),
                        'message': 'Произошла ошибка при запуске согласования в OK'
                    },
                )
            response.raise_for_status()

        ok_id = response.json()['uuid']
        oebs_agreement.ok_id = ok_id
        oebs_agreement.save(update_fields=['ok_id'])
        return ok_id

    def close_request(self, oebs_agreement):
        url = urljoin(self.host, f'api/approvements/{oebs_agreement.ok_id}/close/')

        with Session(oauth_token=self.token) as session:
            response = session.post(url=url)
            response.raise_for_status()

    def add_request_to_ticket(self, oebs_agreement):
        body = (f'{{{{iframe src="{self.host}approvements/{oebs_agreement.ok_id}'
                f'?_embedded=1" frameborder=0 width=100% height=400px '
                f'scrolling=no}}}}')
        return create_comment(oebs_agreement.issue, body)

    def get_request_state(self, oebs_agreement) -> Tuple[str, str]:
        url = urljoin(
            self.host,
            f'api/approvements/{oebs_agreement.ok_id}/'
        )
        with Session(oauth_token=self.token) as session:
            response = session.get(url=url)
            response.raise_for_status()

        response_json = response.json()
        return response_json['status'], response_json['resolution']
