from django.db.models import Q
from django.db import connection


def wrap_in_Q(prefix='', **kwargs):
    """
    Сформировать Q-объект на основе kwargs с возможностью добавления префикса.
    @param prefix: префикс, добавляемый ко всем ключам лукапа.
    @param kwargs: лукап.
    @return: django.db.models.Q
    """
    lookup = kwargs
    if prefix:
        lookup = dict(
            (prefix + '__' + key, val)
            for key, val in lookup.items()
        )
    return Q(**lookup)


def filter_or_none(queryset, query):
    """
    Отфильтровать queryset с помощью Q-объекта query, если query == Q(), делаем
    queryset.none(), чтобы не отфильтровать все сущности в queryset.
    Это помогает избежать частой ошибки, когда одна функция формирует Q,
    который может быть пустым, а во второй функции он используется и вместо
    ожидаемого поведения, что в результате 0 сущностей, получаем всех.
    @param queryset: QuerySet
    @param query: Q
    @return: QuerySet
    """
    if not query:
        return queryset.none()
    else:
        return queryset.filter(query)


def join_prefixes(*prefixes):
    """
    Слепить список префиксов для запросов в один.
    """
    return '__'.join(filter(bool, prefixes))


def qualified_field_name(field):
    table_name = field.model._meta.db_table
    return '.'.join(map(connection.ops.quote_name, (table_name, field.name)))
