"""
Хелперы для django-rest-framework http://www.django-rest-framework.org/
"""

from django.utils.translation import ugettext_lazy as _
from rest_framework import serializers
from rest_framework.exceptions import APIException
from rest_framework.response import Response

from plan.api import validators
from plan.api.exceptions.handler import old_abc_exception_handler
from plan.services.models import Service
from plan.staff.models import Department


class PlanSerializer(serializers.Serializer):
    """
    yawf рассчитывает, что у формы есть cleaned_data, так как мы хотим
    подменить django-формы на сериализаторы, придется эмулировать интерфейс.
    """
    @property
    def cleaned_data(self):
        return self.validated_data


class LocaleSerializer(PlanSerializer):
    """Нужен для парсинга локализованных значений,
    приходящих с фронтенда. Пример: теперь имя сервиса будет
    приходить с фронтенда abc:
        'name': {'ru': 'квекве', 'en': 'qweqwe'}
    """
    ru = serializers.CharField()
    en = serializers.CharField()


class LocaleServiceNameSerializer(PlanSerializer):
    ru = serializers.CharField(validators=[
        validators.UniqueFieldValidator(
            queryset=Service.objects.alive(),
            field='name',
            lookup='iexact',
            message=_('Имя сервиса должно быть уникальным'),
        ),
        validators.UniqueFieldValidator(
            queryset=Department.objects.all(),
            field='name',
            lookup='iexact',
            message=_('Названия департаментов и сервисов не могут пересекаться'),
        ),
    ])
    en = serializers.CharField(validators=[
        validators.UniqueFieldValidator(
            queryset=Service.objects.alive(),
            field='name_en',
            lookup='iexact',
            message=_('Английское имя сервиса должно быть уникальным'),
        ),
        validators.UniqueFieldValidator(
            queryset=Department.objects.all(),
            field='name_en',
            lookup='iexact',
            message=_('Английские названия департаментов и сервисов не могут пересекаться'),
        ),
    ])


def message_serializer_wrapper(serializer_cls):
    """
    Так как yawf инициализирует класс сериализатора с первым позиционным
    аргументом, а нужно передать параметры во второй позиционный
    или в ключевой data. Это и делает эта функция.
    """
    from functools import wraps

    @wraps(serializer_cls)
    def wrapper(raw_params):
        return serializer_cls(data=raw_params)
    return wrapper


def plan_response(content=None, error=None):
    """
    Возвращает Response в формате UniversalView, нужен для переписыванных ручек
    из dehydrate
    """
    if content and error:
        raise Exception("`content` and `error` are mutually exclusive")

    if content is None and error is None:
        raise Exception("`content` or `error` should be given")

    if error:
        status = error.http_code
        context = {
            'content': {},
            'error': {
                'code': error.error_code,
                'message': getattr(error, 'message', str(error)),
                'params': error.params,
            },
        }
    else:
        status = 200
        context = {
            'content': content,
            'error': {},
        }

    return Response(context, status=status)


def call_old_exception(exc):
    """ Приводим параметры к старому виду """
    # TODO: удалить после перехода на новые ручки

    try:
        raise exc
    except APIException as e:
        return old_abc_exception_handler(e, e.get_full_details())
    except Exception:
        raise
