from collections import OrderedDict


class TreeNode(object):
    def __init__(self, id, object=None, skipped=False):
        self.id = id
        self.object = object
        self.children = []
        self.parent = None
        self.skipped = skipped

    def __repr__(self):
        return "<TreeNode: id=%s, object=%s>" % (self.id, self.object)


def filter_subtree(objects, node):
    return [obj for obj in objects if node == obj or node.is_ancestor_of(obj)]


def build_object_map(objects, queryset, load_skipped=False, branch=None, root=None):
    ancestors = queryset.model._closure_model.objects.filter(child__in=objects)

    if root:
        ancestors = ancestors.filter(parent__in=root.get_descendants())
    ancestors_ids = set(ancestors.values_list('parent', flat=True))
    ancestors_ids.update({object.id for object in objects})

    if ancestors_ids:
        objects = queryset.filter(id__in=ancestors_ids)
        if not load_skipped:
            objects = objects.only('id', 'parent__id', 'name', 'owner')

        if branch:
            objects = filter_subtree(objects, branch)
    else:
        return {}

    return dict([(obj.id, obj) for obj in objects])


def build_restored_tree(objects, object_map):
    def insert(node):
        node_map[node.id] = node

        parent_id = None
        if node.id in object_map:
            parent_id = object_map[node.id].parent_id

        if parent_id and parent_id in object_map:
            if parent_id in node_map:
                parent_node = node_map[parent_id]
            else:
                parent_node = TreeNode(parent_id, object_map[parent_id],
                                       skipped=True)
                insert(parent_node)
            parent_node.children.append(node)
            node.parent = parent_node
        else:
            tree.append(node)

    objects.sort(key=lambda o: o.level)

    tree = []
    node_map = {}
    for obj in objects:
        insert(TreeNode(obj.id, obj))

    return tree


class RestoredTreeConverter(object):
    """
    Получает на вход список вершин восстановленных деревьев,
    узлами которых являются сущности TreeNode,
    и отдает представление этого дерева в виде некоего списка, в котором
    элемент - это пара:
    объект и флаг, востанновленный это объект в дереве или нет.

    Конвертация происходит в два этапа:

    1. Группировка вершин дерева.
    Группой является список проектов, которые представляют из себя
    кусок дерева без "дырок".
    Ключ группы - общий родитель.
    Причем корень дерева будет в группе, где ключ - он сам.

    2. Восстановление родителей.
    Для каждого ключа группы восстанавливаем путь до корня.
    Сам корень может быть восстановлен только один раз.

    Например, если дерево такое:
    (1, False)
        (2, True)
        (3, True)
            (4, True)
            (5, False)
                (6, True)
        (7, False)
            (8, False)
                (9, True)
            (10, True)
    (11, False)
        (12, True)
        (13, False)
            (14, True)

    То итоговый список будет таким (вместо id инстансы объектов):
    (1, False)
    (2, True)
    (3, True)
    (4, True)
    (3, False)
    (5, False)
    (6, True)
    (7, False)
    (10, True)
    (7, False)
    (8, False)
    (9, True)
    (11, False)
    (12, True)
    (13, False)
    (14, True)
    """
    def __init__(self, roots):
        self.roots = roots
        self.node_group_map = {}
        self.groups = OrderedDict()
        self.parent_map = {}

    def group(self):
        for root in self.roots:
            self.group_tree(root)

    def group_tree(self, node):
        # TODO: для каждого конвертера, parent_map строится заново,
        # хотя многие проекты будут повторятся
        self.parent_map[node.object.id] = (
            node.parent.object if node.parent else None
        )
        if not node.skipped:
            group = self.detect_group(node)
            self.node_group_map[node.object] = group
            if group not in self.groups:
                self.groups[group] = []
            self.groups[group].append(node.object)

        for child in node.children:
            self.group_tree(child)

    def detect_group(self, node):
        parent = node.parent
        if not parent:
            return node.object

        parent_object = parent.object
        if parent_object in self.node_group_map:
            return self.node_group_map[parent_object]
        return parent_object

    def convert(self):
        self.group()

        converted = []
        self.marked_roots = set()

        for group_parent, subtree in self.groups.items():
            if group_parent != subtree[0]:
                breadcrumbs = self.get_breadcrumbs(group_parent)
                converted.extend((node, False) for node in breadcrumbs)
            else:
                self.marked_roots.add(group_parent)

            converted.extend((node, True) for node in subtree)

        return converted

    def get_breadcrumbs(self, obj):
        breadcrumbs = []
        while True:
            parent = self.parent_map[obj.id]
            if parent:
                breadcrumbs.append(obj)
                obj = parent
            elif obj not in self.marked_roots:
                breadcrumbs.append(obj)
                self.marked_roots.add(obj)
            else:
                break

        return breadcrumbs[::-1]
