import django.contrib.postgres.search
from django.db import connection


def generate_resources_index():
    index_name = "typeid_answerid_unique"
    forward_sql = """
        CREATE UNIQUE INDEX %s
        ON resources_resource (type_id, answer_id)
        WHERE obsolete_id IS NULL AND answer_id IS NOT NULL;
    """
    backward_sql = "DROP INDEX %s;"

    def forward():
        with connection.cursor() as cursor:
            cursor.execute(forward_sql % index_name)

    def backward():
        with connection.cursor() as cursor:
            cursor.execute(backward_sql % index_name)

    return forward, backward


class PrefixSearchQuery(django.contrib.postgres.search.SearchQuery):
    """
    Оборачивает поисковый запрос с заданной конфигурацией в to_tsquery.

    Запрос разбивается по пробельным символам на элементы, по которым осуществляется конъюнктивный префиксный поиск.
    """

    def as_sql(self, compiler, connection):
        params = [self.value]
        if self.config:
            config_sql, config_params = compiler.compile(self.config)
            template = 'to_tsquery({}::regconfig, %s)'.format(config_sql)
            params = config_params + [' & '.join(f'{s.replace(":", "&")}:*' for s in self.value.split())]
        else:
            template = 'plainto_tsquery(%s)'
        if self.invert:
            template = '!!({})'.format(template)
        return template, params
