import logging
import re
from urllib.parse import parse_qsl

from django.conf import settings
from django.utils.encoding import force_text, force_str
from ids.exceptions import BackendError
from ids.registry import registry

from plan.exceptions import IDSRepositoryError

log = logging.getLogger(__name__)

LSR_QUEUE = 'LSR'


def _get_repo(collection):
    try:
        return registry.get_repository(
            'startrek2',
            collection,
            user_agent=settings.ABC_IDS_USER_AGENT,
            oauth_token=settings.OAUTH_ROBOT_TOKEN,
            retries=settings.STARTREK_RETRIES,
            retry_on_status=bool(settings.STARTREK_RETRY_STATUSES),
            statuses_to_retry=settings.STARTREK_RETRY_STATUSES,
        )

    except Exception as exc:
        logging.exception('Startrek error')
        raise IDSRepositoryError(getattr(exc, 'message', str(exc)))


def create_issue(queue, **params):
    repo = _get_repo('issues')
    issue_type = params.pop('type', 'task')
    return repo.create(
        type=issue_type,
        queue=queue,
        **params
    )


def update_issue(key, **params):
    repo = _get_repo('issues')
    issue = repo.client.issues[key]
    repo.update(issue, **params)


def close_issue(key, comment):
    repo = _get_repo('issues')
    issue = repo.client.issues[key]
    if issue.status.key != 'closed':
        for transition in issue.transitions:
            if transition.id in ('close', 'closed'):
                transition.execute(comment=comment, resolution='fixed')
                break
        else:
            log.warning(f'No closing transition for {key}')


def change_state(key, transition):
    repo = _get_repo('issues')
    issue = repo.client.issues[key]
    if issue.status.key == transition:
        log.info(f'Skipping change state for {key}, already {transition}')
        return
    for transition_operation in issue.transitions:
        if transition_operation.id == transition:
            transition_operation.execute()
            break
    else:
        log.warning(f'No such transition {transition} for {key}')


def create_comment(key, comment_text, summonees=None):
    repo = _get_repo('issues')
    issue = repo.client.issues[key]
    issue.comments.create(
        text=comment_text,
        summonees=summonees,
    )


def is_queue_in_startrek(queue):
    queue_repo = _get_repo('queues')

    try:
        queue_repo.get({'id': queue})
    except BackendError as error:
        if error.status_code == 404:
            return False
        elif error.status_code == 403:
            return True
        else:
            raise IDSRepositoryError(getattr(error, 'message', str(error)))
    except Exception as exc:
        logging.exception('Startrek error')
        raise IDSRepositoryError(getattr(exc, 'message', str(exc)))
    return True


def get_tracker_type_by_queue(queue):
    if is_queue_in_startrek(queue):
        return 'startrek'


def get_lsr_amount(service):
    log.info('Try to get LSR tickets amount for %r', service)

    repo = _get_repo('issues')

    filter = {
        'queue': LSR_QUEUE,
        'abcService': service.pk,
    }
    amount = len(repo.client.issues.find(filter=filter))

    log.info('Found %s tickets for LSR <%s>', amount, filter)
    return amount


def get_filter_amount(filter):
    log.info('Try to get tickets amount for filter <%s>', filter)

    repo = _get_repo('issues')

    params = _parse_filter(filter)

    if params:
        amount = len(repo.client.issues.find(**params))

        log.info('Found %s tickets for filter <%s>', amount, params)
        return amount


def _parse_filter(filter_string):
    params = {}

    if '/filters/' not in filter_string:
        log.warning('Not a filter <%s>', filter_string)
        return params

    if '?' in filter_string:
        _, qs = filter_string.split('?', 1)
        qs = force_str(qs)
        for k, v in parse_qsl(qs):
            v = force_text(v)

            if '|' in v:
                v = v.split('|')

            if k == 'query':
                params[k] = v
            else:
                params.setdefault('filter', {})[k] = v

    match = re.search(r'filter:(\d+)', filter_string)
    if match:
        params['filter_id'] = int(match.group(1))

    if params == {}:
        log.warning('Cannot parse filter <%s>', filter_string)

    return params
