import re

from itertools import islice


def camelcase_to_underscore(camelcased_string):
    """
    >>> camelcase_to_underscore('CamelCase')
    'camel_case'
    >>> camelcase_to_underscore('CamelCamelCase')
    'camel_camel_case'
    >>> camelcase_to_underscore('Camel2Camel2Case')
    'camel2_camel2_case'
    >>> camelcase_to_underscore('getHTTPResponseCode')
    'get_http_response_code'
    >>> camelcase_to_underscore('get2HTTPResponseCode')
    'get2_http_response_code'
    >>> camelcase_to_underscore('HTTPResponseCode')
    'http_response_code'
    >>> camelcase_to_underscore('HTTPResponseCodeXYZ')
    'http_response_code_xyz'
    """
    first_cap_re = re.compile('(.)([A-Z][a-z]+)')
    all_cap_re = re.compile('([a-z0-9])([A-Z])')

    s1 = first_cap_re.sub(r'\1_\2', camelcased_string)
    return all_cap_re.sub(r'\1_\2', s1).lower()


def underscore_to_camelcase(underscored_string):
    """
    >>> underscore_to_camelcase('camel_case')
    'CamelCase'
    >>> underscore_to_camelcase('camel_camel_case')
    'CamelCamelCase'
    >>> underscore_to_camelcase('camel2_camel2_case')
    'Camel2Camel2Case'
    >>> underscore_to_camelcase('get_http_response_code')
    'GetHttpResponseCode'
    >>> underscore_to_camelcase('get2_http_response_code')
    'Get2HttpResponseCode'
    >>> underscore_to_camelcase('http_response_code')
    'HttpResponseCode'
    >>> underscore_to_camelcase('http_response_code_xyz')
    'HttpResponseCodeXyz'
    """
    return ''.join(x.capitalize() or '_' for x in underscored_string.split('_'))


def fetch_lines(s, lines_count):
    splitted = (
        string.strip() for string in s.split('\n')
        if string
    )
    return list(islice(splitted, lines_count))


if __name__ == '__main__':
    import doctest
    doctest.testmod()
