import functools
import logging

from django.utils import timezone

from plan import settings
from plan.celery_app import app
from plan.common.utils import locks
from plan.unistat.models import TaskMetric
import contextlib

log = logging.getLogger(__name__)


@contextlib.contextmanager
def task_metric(metric_name, send_to_unistat=False):
    start = timezone.now()
    yield
    task_metric, _ = TaskMetric.objects.update_or_create(
        task_name=metric_name,
        defaults={
            'last_success_start': start,
            'last_success_end': timezone.now(),
            'send_to_unistat': send_to_unistat,
        }
    )


def lock_task(f=None, min_lock_time=settings.ABC_MIN_LOCK_TIME, **kwargs):
    """
    Обертка над декоратором celery-таска.

    Если явно не был задан параметр lock=False, задача будет заблокирована
    через yt с ключом lock_key (или полным именем таска по умолчанию).

    Также, если явно не задан параметр ignore_result=False, в оригинальный
    декоратор будет передан параметр ignore_result=True.

    min_lock_time - время в секундах, на которое гарантировано будет взят лок,
    использутся, если таска отрабатывает слишком быстро, по умолчанию 30 сек

    Остальные параметры передаются в оригинальный декоратор без изменений.
    """
    def get_wrapper(f, options):
        last_finished_metric_name = options.pop('last_finished_metric_name', None)

        @functools.wraps(f)
        def wrapper(*args, **kwargs):
            metric_name = f.__name__ if last_finished_metric_name is None else last_finished_metric_name
            with task_metric(metric_name, send_to_unistat=bool(last_finished_metric_name)):
                task_result = f(*args, **kwargs)
            return task_result

        result = wrapper

        lock = options.pop('lock', True)
        lock_key = options.pop('lock_key', None)
        if lock:
            result = locks.lock(lock_key, min_lock_time)(result)

        task_options = {
            'ignore_result': True,
        }
        task_options.update(options)

        result = app.task(**task_options)(result)
        result.original_func = f

        return result

    if f and callable(f):
        return get_wrapper(f, kwargs)
    else:
        def decorator(f):
            return get_wrapper(f, kwargs)
        return decorator


class RetryExponentialDecorator(object):
    """
    Этим декоратором можно оборачивать таски, потому что __getattr__ позволяет
    обращаться к методам типа delay и тд, а обычный декоратор на основе
    функции — нет.
    """

    def __init__(self, func):
        self.func = func
        self.logger = logging.getLogger(func.__module__)
        functools.update_wrapper(self, func)

    def __call__(self, *args, **kwargs):
        try:
            self.func(*args, **kwargs)
        except Exception as ex:
            self.logger.exception('Exception in function ' + self.func.__name__)

            # Задержка перед повторной отправкой письма увеличивается
            # экспоненциально, вплоть до 3 часов
            raise self.func.retry(
                exc=ex,
                countdown=min(45 * 4 ** self.func.request.retries, 180 * 60)
            )

    def __getattr__(self, item):
        return getattr(self.func, item)


retry_exponential = RetryExponentialDecorator


def retry_func(exceptions, func=None, count=10):

    def decorator(func):
        @functools.wraps(func)
        def wrapper(*args, **kwargs):
            retried_count = 0
            while True:
                try:
                    return func(*args, **kwargs)
                except exceptions:
                    if retried_count >= count:
                        raise
                    retried_count += 1
        return wrapper

    if func is None:
        return decorator
    return decorator(func)


def get_last_success_start(task_name):
    try:
        task_stat = TaskMetric.objects.get(task_name__endswith=task_name)
    except TaskMetric.DoesNotExist:
        return None
    return task_stat.last_success_start
