"""
Мини-библиотека для обработки входных данных.
Имеет вспомогательные классы/функции для:
    * предобработки полученных данных
    * валидации (не изменяет данные)
    * обработки (может изменять данные)
"""
from collections import namedtuple

from plan.common.utils.collection import DotAccessedDict
from plan.common.utils.strings import fetch_lines


def preparation_chain(data, preparators, preserve_initial_data=False):
    """
    Принимает данные и список препараторов, которые будут вызваны по очереди.
    В препараторы передаются изначальные данные (data), а также накопленные
    подготовленные данные (prepared).
    В результате вернется prepared, в котором будет все, что заготовили
    препараторы.

    Предполагается, что препараторы не изменяют data и prepared, их задача —
    вернуть какие-то данные на основе data и, если необходимо, того, что уже
    надобавляли другие препараторы в prepared.

    preserve_initial_data = True положит в возвращаемое значение все ключи
    из data перед тем как передавать их в препараторы.

    data для препараторов и result являются словарями, которые поддерживают
    доступ к ключам через атрибуты.
    """
    prepared = DotAccessedDict()
    if preserve_initial_data:
        prepared.update(data)

    wrapped_data = DotAccessedDict(data)

    for preparator in preparators:
        key = preparator.__name__
        if key.startswith('prepare_'):
            key = key[len('prepare_'):]
        prepared[key] = preparator(wrapped_data, prepared)

    return prepared


ValidationError = namedtuple('ValidationError', 'code message params')


def validation_chain(data, validators, accumulate=False):
    """
    Принимает данные и список валидаторов, по очереди скармливает данные
    каждому валидатору. В зависмости от параметра accumulate можно накапливать
    ошибки или вернуть первую.

    Возвращает имя валидатора и ошибку ValidationError, которую он вернул.

    Если имя валидатора начинается на validate_, то префикс отбрасывается для
    формирования ключей в итоговых ошибках.
    Код ошибки и message берутся из первой и второй непустой строки докстринга
    валидатора соответственно.
    """
    errors = {}

    for validator in validators:
        error_params = validator(data)
        if not error_params:
            continue

        key = validator.__name__
        if key.startswith('validate_'):
            key = key[len('validate_'):]

        code, message = fetch_lines(validator.__doc__, lines_count=2)

        error = ValidationError(
            code=code,
            message=message,
            params=error_params,
        )
        if not accumulate:
            return key, error
        errors[key] = error

    return errors


def process_chain(data, procesors):
    """
    По очереди передает данные по конвейеру, каждый процессор может сделать
    с данными что угодно и вернуть, что угодно, поэтому процессоры
    должны быть совместимы.
    """
    result = data
    for processor in procesors:
        result = processor(result)
    return result
