from typing import Optional
from urllib.parse import urljoin
from django.conf import settings

from plan.common.utils.http import Session
from plan.common.utils.tvm import get_tvm_ticket


class WatcherClient:
    def __init__(self):
        self.host = settings.WATCHER_API_URL

    def _get_headers(self):
        return {
            'X-Ya-Service-Ticket': get_tvm_ticket(destination=str(settings.WATCHER_TVM_ID)),
            'X-UID': '1120000000015421', #  zomb-prj-282
        }

    def get_response(self, url, filter_params, field_params=None, params=None) -> list:
        has_next = True
        initial = True
        result = []
        if params is None:
            params  = {}
        params['page_size'] = 1000

        if filter_params:
            params['filter'] = ','.join(filter_params)
        if field_params:
            params['fields'] = ','.join(field_params)
        while has_next:
            with Session() as session:
                response = session.get(
                    url=url,
                    headers=self._get_headers(),
                    params=params if initial else None,
                )
                response.raise_for_status()
            response_data = response.json()
            result.extend(response_data['result'])
            url = response_data.get('next')
            initial = False
            if not url:
                has_next = False
        return result

    def get_shifts(
        self, service_id: Optional[str] = None,
        schedule_id: Optional[str] = None,
        schedule_slug: Optional[str] = None,
        service_slug: Optional[str] = None,
        date_to: Optional[str] = None,
        date_from: Optional[str] = None,
        current: Optional[bool] = None,
    ) -> list:
        url = urljoin(self.host, 'v1/shift/for_abc')
        filter_params = ['empty=False', 'schedule.state=active']
        params = {}
        if current:
            params['current'] = current
        if service_id:
            filter_params.append(f'schedule.service_id={service_id}')
        if schedule_id:
            filter_params.append(f'schedule_id={schedule_id}')
        if service_slug:
            filter_params.append(f'schedule.service.slug={service_slug}')
        if schedule_slug:
            filter_params.append(f'schedule.slug={schedule_slug}')
        if date_to:
            filter_params.append(f'start__lte={date_to}T23:59:59')
        if date_from:
            filter_params.append(f'end__gte={date_from}')

        return self.get_response(url=url, filter_params=filter_params, params=params)

    def get_schedules(
        self, service_id: Optional[str] = None,
        state: Optional[str] = None,
        schedule_id: Optional[str] = None,
        service_slug: Optional[str] = None,
        field_params: Optional[list] = None,
    ) -> list:
        url = urljoin(self.host, 'v1/schedule/')
        filter_params = []
        if service_id:
            filter_params.append(f'service_id={service_id}')
        if service_slug:
            filter_params.append(f'service.slug={service_slug}')
        if schedule_id:
            filter_params.append(f'id={schedule_id}')
        if state:
            filter_params.append(f'state={state}')

        return self.get_response(url=url, filter_params=filter_params, field_params=field_params)
